/*********************************************************************
 * MintCan2 DCC/MM Controler r0.1
 *
 * Copyright (C) 2018 twaydcc
 *
 */

#include <arduino.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <EEPROM.h>
#include <SPI.h>

#include "DSCoreM_Type.h"
#include "DSCoreM_Common.h"
#include "DSCoreM_List.h"
#include "DSCoreM_DCC.h"
#include "DSCoreM_MM2.h"
#include "DSCoreM.h"
#include "OzOLED_spi.h"

// Definition constance

//#define _DEBUG
#define _SPEEDBAR

#define BUTTON_SEL  0b00000100
#define BUTTON_UP   0b00001000
#define BUTTON_DOWN 0b00010000
#define BUTTON_F1   0b00100000
#define BUTTON_F2   0b01000000
#define BUTTON_DIR  0b10000000

#define MODE_CV		1
#define MODE_LADR	2
#define MODE_LOC	3

#define DRV_LOC		0
#define DRV_FN1		1
#define DRV_FN2		2
#define DRV_ACC		3

#define MAX_LOCO	51
#define MAX_FUNC	28
#define MAX_ACC		99

#define DC_PIN		16
#define RST_PIN		15
#define CS_PIN		8

#define CONTINUOUS	4

#define EEPOFFSET	8

// Class
DSCoreLib DSCore;
OzOLED LCD(DC_PIN, RST_PIN, CS_PIN);
char gText[4 + 1];

//Loc mode
byte gAddrIdx;     // LocAddress index
byte gAddrIdxOld;
word gLocAddress;

byte gColValue[4];
//byte gConsAddress = 6;
//byte gEStop = 0;

boolean bAnalog = false;

byte gButton_val = 0;
byte gButton_old = 0;

byte gMode_Main = MODE_LOC;
byte gMode_Drive = DRV_LOC;

//Protcol
byte eepProtcol = 0;
word gLocProtcol = ADDR_DCC;
word gAccProtcol = ADDR_ACC_DCC;

//Func
byte gFunc1 = 0;
byte gFunc1Old;
byte gFunc2 = 1;
byte gFunc2Old;
byte gLocFunc1 = 0;
byte gLocFunc2 = 0;


//Acc mode
word gAccAddress = 1;
word gAccAddressOld;

unsigned long gTime_100ms = 0;
byte gPressCount = 0;

// Functions
void displayInit();
void displayModeCV();
void displayModeLongAddr();
void displayModeLoc();
void displayStepIcon(byte inSpeed);
void processModeCV();
void processModeLongAddr();
void processModeDial();
void processModeSel();
void processModeLoc();
void processModeFn();
void processModeAcc();
byte getFnIndex(byte inFunc, byte *inBit);
byte getButtonStatus();
bool checkButton(byte inBuf, byte inCurrent, byte inButtonBit);
bool checkButtonLeaved(byte inBuf, byte inCurrent, byte inButtonBit);
bool checkButtonContinuous(byte inBuf, byte inCurrent, byte inButtonBit);
byte loadEEPROM();
void loadLongAddress(byte inAdrIdx); //load gLocAddress
void saveLongAddress(byte inAdrIdx);
void saveEEPROM(byte inMode);


/*******************************
    Setup
********************************/

void setup()
{
	pinMode(10, OUTPUT);
	digitalWrite(10, LOW);
	pinMode(9, OUTPUT);
	digitalWrite(9, LOW);
	
	//Serial
	#ifdef _DEBUG
	Serial.begin(115200);
	while (!Serial);

	Serial.println("-----------------------");
	Serial.println("twaydcc MintCan4 rev0.1");
	Serial.println("-----------------------");
	Serial.println("100 Ready");
	#endif
	
	//OLDE init
	displayInit();
	
	//Analog input
	pinMode(A0, INPUT);
	digitalWrite(A0, LOW); // disable internal pullup
	analogReference(DEFAULT); // set analog reference 5V(VCC) 
	
	//Faster Analog read
	ADCSRA = ADCSRA & 0xf8;
	ADCSRA = ADCSRA | 0x04;
	
	//Function button
	pinMode(2, INPUT);
	pinMode(3, INPUT);
	pinMode(4, INPUT);
	pinMode(5, INPUT);
	pinMode(6, INPUT);
	//Direction button
	pinMode(7, INPUT);
	// input pullup
	digitalWrite(2, HIGH);
	digitalWrite(3, HIGH);
	digitalWrite(4, HIGH);
	digitalWrite(5, HIGH);
	digitalWrite(6, HIGH);
	digitalWrite(7, HIGH);
	
	#ifdef _DEBUG
	pinMode(A3, OUTPUT);
	//digitalWrite(A3, HIGH);
	#endif
	
	//Load EEPROM
	//EEPROM.write(0, 0xff);
	gAddrIdx = loadEEPROM();
	gAddrIdxOld = gAddrIdx;
	
	loadLongAddress(gAddrIdx);
	gColValue[0] = gLocAddress / 1000;
	byte mod = gLocAddress % 1000;
	gColValue[1] = mod / 100;
	mod = mod % 100;
	gColValue[2] = mod / 10;
	gColValue[3] = mod % 10;
	
	gFunc1Old = gFunc1;
	gFunc2Old = gFunc2;
	gAccAddressOld = gAccAddress;
	
	#ifdef _DEBUG
	Serial.print("Mode:");
	Serial.println(gMode_Drive);
	Serial.print("Cur LocAddr:");
	Serial.print(gLocAddress);
	Serial.print("[");
	Serial.print(gAddrIdx);
	Serial.println("]");
	#endif
	
	//Set default button status
	gButton_val = getButtonStatus();
	
	//mode check
	if((gButton_val & BUTTON_F2) > 0) {
		gMode_Main = MODE_CV;
		
		#ifdef _DEBUG
		Serial.println("<<CV write mode>>");
		#endif
	} else if((gButton_val & BUTTON_F1) > 0) {
		gMode_Main = MODE_LADR;
		
		#ifdef _DEBUG
		Serial.println("<<Long address mode>>");
		#endif
	} else if((gButton_val & BUTTON_UP) > 0) {
		// Change to DCC mode
		eepProtcol = 0;
		
		//Update Protocol
		saveEEPROM(1);
		
		#ifdef _DEBUG
		Serial.println("<<Change to DCC mode>>");
		#endif
	} else if((gButton_val & BUTTON_DOWN) > 0) {
		// Change to MM2 mode
		eepProtcol = 1;
		
		//Update Protocol
		saveEEPROM(1);
		
		#ifdef _DEBUG
		Serial.println("<<Change to MM2 mode>>");
		#endif
	}
	
	gButton_old = gButton_val;
	
	/* Protcol changer */
	if( eepProtcol == 1)
	{
		/* MM2 Mode */
		//Serial.println("MM2 Mode");
		gLocProtcol = ADDR_MM2;
		gAccProtcol = ADDR_ACC_MM2;
	}
	else
	{
		/* DCC Mode */
		//Serial.println("DCC Mode");
		gLocProtcol = ADDR_DCC;
		gAccProtcol = ADDR_ACC_DCC - 1;
	}
	
	//Init DSCore
	DSCore.Init();
	
	//LCD
	switch( gMode_Main) {
		case MODE_CV:
			displayModeCV();
			break;
		case MODE_LOC:
			displayModeLoc();
			break;
		case MODE_LADR:
			displayModeLongAddr();
			break;
	}
	
	while ((gButton_val & BUTTON_UP) > 0 || (gButton_val & BUTTON_DOWN) > 0)
	{
		gButton_val = getButtonStatus();
		gPressCount = 0;
	}
	
	gTime_100ms = millis();
	
	#ifdef _DEBUG
	Serial.println("Done");
	//digitalWrite(A3, LOW);
	#endif
}

/*******************************
    Main loop
********************************/

void loop()
{
	//Scan
	if( (gLocAddress != 0) && (gMode_Main != MODE_CV))
	{
		//Power on
		if( DSCore.IsPower() == false)
		{
			DSCore.SetPower(true);
		}
		
		DSCore.Scan();
	}
	
	// 100ms interval
	if( (millis() - gTime_100ms) >= 100)
	{
		gButton_val = getButtonStatus();
		
		switch( gMode_Main)
		{
			case MODE_CV:
				processModeCV();
				break;
				
			case MODE_LADR:
				processModeLongAddr();
				break;
				
			case MODE_LOC:
				//Update dial for speed control
				processModeDial();
				
				if( gLocAddress > 0)
				{
					processModeSel();
				}
				
				if( gMode_Drive == DRV_LOC)
				{
					processModeLoc();
				}
				else if(gMode_Drive == DRV_ACC)
				{
					processModeAcc();
				}
				else
				{
					processModeFn();
				}
				break;
		}
		
		if(gButton_val != gButton_old)
		{
			gButton_old = gButton_val;
		}
		
		//Reset task
		gTime_100ms = millis();
	}
}

/*******************************
    OLED init
********************************/

void displayInit()
{
	LCD.init();
	LCD.clearDisplay();           //clear the screen and set start position to top left corner
	LCD.setNormalDisplay();       //Set display to Normal mode
	LCD.setPageMode();            //Set addressing mode to Page Mode
}

/*******************************
    Display CV mode
********************************/

void displayModeCV()
{
	LCD.printString("<CV programmer>", 2, 0);
	//CV No.
	LCD.printString("CV No", 3, 2);
	LCD.drawIcon(10, 2, 9, false);
	LCD.drawIcon(10, 3, 10, false);
	sprintf(gText, "%2d", 1);
	LCD.printBigNumber(gText, 13, 2);
	//CV value
	LCD.printString("CV Val", 3, 4);
	sprintf(gText, "%3d", 0);
	LCD.printBigNumber(gText, 12, 4);
	//Button
	LCD.printStringNOT("SEL", 0, 7);
}

/*******************************
    Display Long address mode
********************************/

void displayModeLongAddr()
{
	byte mod=0;
	LCD.printString("<Long Address>", 2, 0);
	// Source address
	LCD.printString("Src ", 3, 2);
	LCD.drawIcon(8, 2, 9, false);
	LCD.drawIcon(8, 3, 10, false);
	sprintf(gText, "%2d", gAddrIdx);
	LCD.printBigNumber(gText, 12, 2);
	
	// Distnation address
	LCD.printString("Dist", 3, 4);
	// col4
	sprintf(gText, "%1d", gLocAddress / 1000);
	LCD.printBigNumber(gText, 10, 4);
	// col3
	mod = gLocAddress % 1000;
	sprintf(gText, "%1d", mod / 100);
	LCD.printBigNumber(gText, 11, 4);
	// col2
	mod = mod % 100;
	sprintf(gText, "%1d", mod / 10);
	LCD.printBigNumber(gText, 12, 4);
	// col1
	sprintf(gText, "%1d", mod % 10);
	LCD.printBigNumber(gText, 13, 4);
	
	//Button
	LCD.printStringNOT("SEL", 0, 7);
}

/*******************************
    Display Loco mode
********************************/

void displayModeLoc()
{
	// Loco
	LCD.printString("Loc", 0, 0);
	LCD.drawIcon(4, 0, 9, false);
	LCD.drawIcon(4, 1, 10, false);
	// Protcol
	if(gLocAddress == 0) {
		LCD.printString("Ana", 14, 0);
	} else {
		if(gLocProtcol == ADDR_DCC) {
			LCD.printString("DCC", 14, 0);
		} else {
			LCD.printString("MM ", 14, 0);
		}
	}
	// Loco address
	sprintf(gText, "%4d", gLocAddress);
	LCD.printBigNumber(gText, 5, 0);
	
	// display speed
	#ifdef _SPEEDBAR
	LCD.drawSpeedBar( 14, 1, 0, false, false);
	#else
	sprintf(gText, "%3d", 0);
	LCD.printString(gText, 14, 1);
	#endif
	
	// Direction
	LCD.printString(" ", 13, 1);
	LCD.drawIcon(17, 1, 11, false); // Fwd
	
	// Function
	LCD.printString("Fn", 0, 2);
	sprintf(gText, "%2d", gFunc1);
	LCD.printBigNumber(gText, 5, 2);
	// Fn button
	LCD.printString("Off", 7, 3);
	LCD.printStringNOT("F", 11, 7);
	LCD.printStringNOT(gText, 12, 7);

	sprintf(gText, "%2d", gFunc2);
	LCD.printBigNumber(gText, 12, 2);
	// Fx button
	LCD.printString("Off", 14, 3);
	LCD.printStringNOT("F", 15, 7);
	LCD.printStringNOT(gText,16, 7);
	
	// Accesory
	LCD.printString("Acc", 0, 4);
	sprintf(gText, "%2d", gAccAddress);
	LCD.printBigNumber(gText, 5, 4);

	LCD.printString("Div", 7, 5);
	LCD.printString("/", 10, 5);
	
	//Display Button
	LCD.printStringNOT("SEL", 0, 7);
}

/*******************************
    processModeCV
********************************/

void processModeCV()
{
	static byte aMode_CV = 0;  // 0:addr, 1:value
	static byte aCVAddress = 1;
	static byte aCVValue = 0;
	
	if(gButton_val != gButton_old)
	{
		// select CVAddress or CVValue
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			gPressCount = 0;
			
			switch( aMode_CV)
			{
			case 0: //CV No select
				//Increment CV no
				aCVAddress++;
				if(aCVAddress > MAX_ACC) {
					aCVAddress = 1;
				}
				
				//Display CV No.
				sprintf(gText, "%2d", aCVAddress);
				LCD.printBigNumber(gText, 13, 2);
				break;
				
			case 1: //CV Value select
				//Increment CV no
				aCVValue++;
				if(aCVValue > 255) {
					aCVValue = 0;
				}
				
				//Display CV Val.
				sprintf(gText, "%3d",aCVValue);
				LCD.printBigNumber(gText, 12, 4);
				break;
			}
			
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			gPressCount = 0;
			
			switch( aMode_CV)
			{
			case 0: //CV No select
				//Decrement CV no
				if(aCVAddress == 1) {
					aCVAddress = MAX_ACC;
				} else {
					//Decrement address
					aCVAddress--;
				}
				
				//Display CV No.
				sprintf(gText, "%2d", aCVAddress);
				LCD.printBigNumber(gText, 13, 2);
				break;
				
			case 1: //CV Value select
				//Decrement CV no
				if(aCVValue == 0) {
					aCVValue = 255;
				} else {
					//Decrement address
					aCVValue--;
				}
				
				//Display CV Val.
				sprintf(gText, "%3d",aCVValue);
				LCD.printBigNumber(gText, 12, 4);
			}
		}
		
		// select CVAddress, CVValue
		else if (checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			//モード切替
			if( aMode_CV == 1)
			{
				aMode_CV = 0;
				
				LCD.printString(" ", 10, 4);
				LCD.printString(" ", 10, 5);
				LCD.drawIcon(10, 2, 9, false);
				LCD.drawIcon(10, 3, 10, false);
				LCD.printString("   ", 15, 7);
			}
			else
			{
				aMode_CV = 1;
				
				LCD.printString(" ", 10, 2);
				LCD.printString(" ", 10, 3);
				LCD.drawIcon(10, 4, 9, false);
				LCD.drawIcon(10, 5, 10, false);
				LCD.printStringNOT("ENT", 15, 7);
			}
		}
		
		// Set CVValue
		else if (checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if(aMode_CV == 1)
			{
				LCD.printString("****", 11, 6);
				
				#ifdef _DEBUG
				Serial.print("CV Write No=");
				Serial.print(aCVAddress);
				Serial.print(", Val=");
				Serial.println(aCVValue);
				Serial.println("CV Writing...");
				#endif
				
				//CV Write
				DSCore.WriteConfig_Dir(aCVAddress, aCVValue);
				
				#ifdef _DEBUG
				Serial.println("CV Write finished!");
				#endif
				
				LCD.printString("    ", 11, 6);
				
				//Mode
				LCD.printString(" ", 10, 4);
				LCD.printString(" ", 10, 5);
				LCD.drawIcon(10, 2, 9, false);
				LCD.drawIcon(10, 3, 10, false);
				
				aMode_CV = 0;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				switch( aMode_CV)
				{
				case 0: //CV No select
					//Increment CV no
					aCVAddress++;
					if(aCVAddress > MAX_ACC) {
						aCVAddress = 1;
					}
					
					//Display CV No.
					sprintf(gText, "%2d", aCVAddress);
					LCD.printBigNumber(gText, 13, 2);
					break;
					
				case 1: //CV Value select
					//Increment CV no
					aCVValue++;
					if(aCVValue > 255) {
						aCVValue = 0;
					}
					
					//Display CV Val.
					sprintf(gText, "%3d",aCVValue);
					LCD.printBigNumber(gText, 12, 4);
					break;
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				switch( aMode_CV)
				{
				case 0: //CV No select
					//Decrement CV no
					if(aCVAddress == 1) {
						aCVAddress = MAX_ACC;
					} else {
						//Decrement address
						aCVAddress--;
					}
					
					//Display CV No.
					sprintf(gText, "%2d", aCVAddress);
					LCD.printBigNumber(gText, 13, 2);
					break;
					
				case 1: //CV Value select
					//Decrement CV no
					if(aCVValue == 0) {
						aCVValue = 255;
					} else {
						//Decrement address
						aCVValue--;
					}
					
					//Display CV Val.
					sprintf(gText, "%3d",aCVValue);
					LCD.printBigNumber(gText, 12, 4);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*******************************
    processMode Long address
********************************/

void processModeLongAddr()
{
	static byte aCol = 0;  // 0:src, 1~4:dist
	static word aDstAddr=0;
	
	if(gButton_val != gButton_old)
	{
		// select Spurce address or Distnation address
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			gPressCount = 0;
			
			//Increment address
			if( aCol == 0)
			{
				//Src select
				gAddrIdx++;
				if(gAddrIdx >= MAX_LOCO) {
					gAddrIdx = 1;
				}
				
				loadLongAddress(gAddrIdx);
				
				//Display Source address
				sprintf(gText, "%2d", gAddrIdx);
				LCD.printBigNumber(gText, 12, 2);
				
				//Display Distnation address
				// col3
				LCD.printString("Dist", 3, 4);
				sprintf(gText, "%1d", gLocAddress / 1000);
				LCD.printBigNumber(gText, 10, 4);
				gColValue[0] = gLocAddress / 1000;
				// col2
				byte mod = gLocAddress % 1000;
				sprintf(gText, "%1d", mod / 100);
				LCD.printBigNumber(gText, 11, 4);
				gColValue[1] = mod / 100;
				// col1
				mod = mod % 100;
				sprintf(gText, "%1d", mod / 10);
				LCD.printBigNumber(gText, 12, 4);
				gColValue[2] = mod / 10;
				
				// col0
				sprintf(gText, "%1d", mod % 10);
				LCD.printBigNumber(gText, 13, 4);
				gColValue[3] = mod % 10;
				
			}
			else
			{
				//Dist select
				gColValue[aCol - 1]++;
				if(gColValue[aCol - 1] > 9)
				{
					gColValue[aCol - 1] = 0;
				}
				
				sprintf(gText, "%1d", gColValue[aCol - 1]);
				LCD.printBigNumber(gText, 9 + aCol, 4);
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			gPressCount = 0;
			
			//Decrement address
			if( aCol == 0)
			{
				//Src select
				if(gAddrIdx == 1) {
					gAddrIdx = MAX_LOCO - 1;
				} else {
					gAddrIdx--;
				}
				
				loadLongAddress(gAddrIdx);
				
				//Display Source address
				sprintf(gText, "%2d", gAddrIdx);
				LCD.printBigNumber(gText, 12, 2);
				//Display Distnation address
				// col3
				LCD.printString("Dist", 3, 4);
				sprintf(gText, "%1d", gLocAddress / 1000);
				LCD.printBigNumber(gText, 10, 4);
				gColValue[0] = gLocAddress / 1000;
				// col2
				byte mod = gLocAddress % 1000;
				sprintf(gText, "%1d", mod / 100);
				LCD.printBigNumber(gText, 11, 4);
				gColValue[1] = mod / 100;
				// col1
				mod = mod % 100;
				sprintf(gText, "%1d", mod / 10);
				LCD.printBigNumber(gText, 12, 4);
				gColValue[2] = mod / 10;
				// col0
				sprintf(gText, "%1d", mod % 10);
				LCD.printBigNumber(gText, 13, 4);
				gColValue[3] = mod % 10;
			}
			else
			{
				if(gColValue[aCol - 1] == 0) {
					gColValue[aCol - 1] = 9;
				} else {
					gColValue[aCol - 1]--;
				}
				
				sprintf(gText, "%1d", gColValue[aCol - 1]);
				LCD.printBigNumber(gText, 9 + aCol, 4);
			}
		}
		
		else if(checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			//モード切替
			aCol++;
			if(aCol > 4) aCol = 0;
			
			if( aCol == 0)
			{
				LCD.printString(" ", 8, 4);
				LCD.printString(" ", 8, 5);
				LCD.printString(" ", 13, 6);
				LCD.drawIcon(8, 2, 9, false);
				LCD.drawIcon(8, 3, 10, false);
				LCD.printString("   ", 15, 6);
				LCD.printString("   ", 15, 7);
			}
			else if(aCol == 1)
			{
				LCD.printString(" ", 8, 2);
				LCD.printString(" ", 8, 3);
				LCD.drawIcon(8, 4, 9, false);
				LCD.drawIcon(8, 5, 10, false);
				LCD.printString("*", 10, 6);
				LCD.printStringNOT("ENT", 15, 7);
			}
			else
			{
				LCD.printString(" ", 8 + aCol, 6);
				LCD.printString("*", 9 + aCol, 6);
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if(aCol > 0)
			{
				LCD.printString("****", 10, 6);
				
				gLocAddress = (gColValue[0] * 1000 + gColValue[1] * 100
								+ gColValue[2] * 10 + gColValue[3]);
				saveLongAddress(gAddrIdx);
				
				aCol = 0;
				
				LCD.printString(" ", 8, 4);
				LCD.printString(" ", 8, 5);
				LCD.drawIcon(8, 2, 9, false);
				LCD.drawIcon(8, 3, 10, false);
				_delay_ms(500);
				LCD.printString("    ", 10, 6);
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment address
				if( aCol == 0)
				{
					//Src select
					gAddrIdx++;
					if(gAddrIdx >= MAX_LOCO) {
						gAddrIdx = 1;
					}
					
					loadLongAddress(gAddrIdx);
					
					//Display Source address
					sprintf(gText, "%2d", gAddrIdx);
					LCD.printBigNumber(gText, 12, 2);
					
					//Display Distnation address
					// col3
					LCD.printString("Dist", 3, 4);
					sprintf(gText, "%1d", gLocAddress / 1000);
					LCD.printBigNumber(gText, 10, 4);
					gColValue[0] = gLocAddress / 1000;
					// col2
					byte mod = gLocAddress % 1000;
					sprintf(gText, "%1d", mod / 100);
					LCD.printBigNumber(gText, 11, 4);
					gColValue[1] = mod / 100;
					// col1
					mod = mod % 100;
					sprintf(gText, "%1d", mod / 10);
					LCD.printBigNumber(gText, 12, 4);
					gColValue[2] = mod / 10;
					
					// col0
					sprintf(gText, "%1d", mod % 10);
					LCD.printBigNumber(gText, 13, 4);
					gColValue[3] = mod % 10;
				}
				else
				{
					//Dist select
					
					gColValue[aCol - 1]++;
					if(gColValue[aCol - 1] > 9)
					{
						gColValue[aCol - 1] = 0;
					}
					
					sprintf(gText, "%1d", gColValue[aCol - 1]);
					LCD.printBigNumber(gText, 9 + aCol, 4);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Decrement address
				if( aCol == 0)
				{
					//Src select
					if(gAddrIdx == 0) {
						gAddrIdx = MAX_LOCO - 1;
					} else {
						gAddrIdx--;
					}
					
					loadLongAddress(gAddrIdx);
					
					//Display Source address
					sprintf(gText, "%2d", gAddrIdx);
					LCD.printBigNumber(gText, 12, 2);
					//Display Distnation address
					// col3
					LCD.printString("Dist", 3, 4);
					sprintf(gText, "%1d", gLocAddress / 1000);
					LCD.printBigNumber(gText, 10, 4);
					gColValue[0] = gLocAddress / 1000;
					// col2
					byte mod = gLocAddress % 1000;
					sprintf(gText, "%1d", mod / 100);
					LCD.printBigNumber(gText, 11, 4);
					gColValue[1] = mod / 100;
					// col1
					mod = mod % 100;
					sprintf(gText, "%1d", mod / 10);
					LCD.printBigNumber(gText, 12, 4);
					gColValue[2] = mod / 10;
					// col0
					sprintf(gText, "%1d", mod % 10);
					LCD.printBigNumber(gText, 13, 4);
					gColValue[3] = mod % 10;
				}
				else
				{
					//Dist select
					
					if(gColValue[aCol - 1] == 0) {
						gColValue[aCol - 1] = 9;
					} else {
						gColValue[aCol - 1]--;
					}
					
					sprintf(gText, "%1d", gColValue[aCol - 1]);
					LCD.printBigNumber(gText, 9 + aCol, 4);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*******************************
    processModeDial
********************************/

void processModeDial()
{
	// Dir
	static byte gLocDirection = GO_FWD;
	
	if(gButton_val != gButton_old)
	{
		if(checkButton(gButton_old, gButton_val, BUTTON_DIR))
		{
			gLocDirection = (gLocDirection == GO_FWD) ? GO_REV : GO_FWD;
			DSCore.SetLocoDirection(gLocProtcol + gLocAddress, gLocDirection);
			
			if (gAddrIdx != gAddrIdxOld)
			{
				//Update LocAddress
				saveEEPROM(0);
				
				gAddrIdxOld = gAddrIdx;
			}
			
			//Display Direction
			if(gLocDirection == GO_FWD)
			{
				LCD.printString(" ", 13, 1);
				LCD.drawIcon(17, 1, 11, false); // Fwd
			}
			else
			{
				LCD.drawIcon(13, 1, 12, false); // Rev
				LCD.printString(" ", 17, 1);
			}
		}
	}
	
	//Dial
	static byte aLocSpeed = 0;
	//static byte aLocStep = 0;
	int aVol_value = (analogRead(A0) & 0b1111111100); // max 1020
	byte aSpeed = aVol_value >> 2;
	byte aStep = aSpeed >> 1;
	
	if (aSpeed != aLocSpeed)
	//if (aStep != aLocSpeed)
	{
		if (gAddrIdx != gAddrIdxOld)
		{
			//Update LocAddress
			saveEEPROM(0);
			
			gAddrIdxOld = gAddrIdx;
		}
		
		// analog
		if(gLocAddress == 0)
		{
			if( gLocDirection == GO_FWD)
			{
				analogWrite(9, 255 - aSpeed);
				analogWrite(10, 255);
			}
			else
			{
				analogWrite(10, 255 - aSpeed);
				analogWrite(9, 255);
			}
			
			// display analog speed
			#ifdef _SPEEDBAR
			LCD.drawSpeedBar( 14, 1, aSpeed, false, false);
			#else
			sprintf(gText, "%3d", aSpeed);
			LCD.printString(gText, 14, 1);
			#endif
			
			#ifdef _DEBUG
			Serial.print("Dial:");
			Serial.println(aSpeed);
			#endif
		}
		
		// digital
		else
		{
			#ifdef _DEBUG
			digitalWrite(A3, HIGH);
			#endif
			
			DSCore.SetLocoSpeedEx(gLocProtcol + gLocAddress, aVol_value, SPEEDSTEP_DCC127);
			
			// display speed
			#ifdef _SPEEDBAR
			LCD.drawSpeedBar( 14, 1, aSpeed, false, false);
			#else
			sprintf(gText, "%3d", aSpeed);
			LCD.printString(gText, 14, 1);
			#endif
			
			#ifdef _DEBUG
			digitalWrite(A3, LOW);
			Serial.print("Dial:");
			Serial.println(aSpeed>>1);
			#endif
		}
		
		aLocSpeed = aSpeed;
		//aLocSpeed = aStep;
	}
}

/*******************************
    processModeSelect
********************************/

void processModeSel()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButtonLeaved(gButton_old, gButton_val, BUTTON_SEL))
		{
			gMode_Drive++;
			if(gMode_Drive > DRV_ACC) {
				gMode_Drive = DRV_LOC;
			}
			
			gPressCount = 0;
			
			switch(gMode_Drive)
			{
				case DRV_LOC:
					LCD.printString(" ", 4, 4);
					LCD.printString(" ", 4, 5);
					LCD.drawIcon(4, 0, 9, false);
					LCD.drawIcon(4, 1, 10, false);
					break;
				case DRV_FN1:
					LCD.printString(" ", 4, 0);
					LCD.printString(" ", 4, 1);
					LCD.drawIcon(4, 2, 9, false);
					LCD.drawIcon(4, 3, 10, false);
					
					break;
				case DRV_FN2:
					LCD.printString(" ", 4, 2);
					LCD.printString(" ", 4, 3);
					LCD.drawIcon(11, 2, 9, false);
					LCD.drawIcon(11, 3, 10, false);
					
					break;
				case DRV_ACC:
					LCD.printString(" ", 11, 2);
					LCD.printString(" ", 11, 3);
					LCD.drawIcon(4, 4, 9, false);
					LCD.drawIcon(4, 5, 10, false);
					break;
			}
			
			//Display Button
			if(gMode_Drive == DRV_ACC)
			{
				LCD.printStringNOT("DIV", 11, 7);
				LCD.printStringNOT("STR", 15, 7);
			}
			else
			{
				LCD.printStringNOT("F", 11, 7);
				sprintf(gText, "%2d", gFunc1);
				LCD.printStringNOT(gText, 12, 7);
				LCD.printStringNOT("F", 15, 7);
				sprintf(gText, "%2d", gFunc2);
				LCD.printStringNOT(gText,16, 7);
			}
		}
	}
}

/*******************************
    processModeLoc
********************************/

void processModeLoc()
{
	static boolean bupdate = false;
	
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			//Increment address
			gAddrIdx++;
			if(gAddrIdx >= MAX_LOCO) {
				gAddrIdx = 0;
				
				TCCR0B &= B11111000; // B1,B2 PWM mode
				TCCR0B |= B00000011; // 1/64(500Hz)
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = true;
			}
			else
			{
				if(bAnalog)
				{
					TCCR0B = 3; // Normal mode, 1/64 prescal
					analogWrite(9, 0);
					analogWrite(10, 0);
					
					bAnalog = false;
				}
			}
			
			loadLongAddress(gAddrIdx);
			
			#ifdef _DEBUG
			Serial.print("ModeLoc:");
			Serial.print(gLocAddress);
			Serial.print("[");
			Serial.print(gAddrIdx);
			Serial.println("]");
			#endif
			
			//Display Loc Address
			sprintf(gText, "%4d", gLocAddress);
			LCD.printBigNumber(gText, 5, 0);
			
			//Display LocProtcol
			if(gAddrIdx == 0)
			{
				LCD.printString("Ana", 14, 0);
				bupdate = false;
			}
			else if(!bupdate)
			{
				if(gLocProtcol == ADDR_DCC) {
					LCD.printString("DCC", 14, 0);
				} else {
					LCD.printString("MM ", 14, 0);
				}
				bupdate = true;
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gAddrIdx == 0)
			{
				gAddrIdx = MAX_LOCO - 1;
			}
			else
			{
				//Decrement address
				gAddrIdx--;
			}
			if (!bAnalog && gAddrIdx == 0)
			{
				TCCR0B &= B11111000; // B1,B2 PWM mode
				TCCR0B |= B00000011; // 1/64(500Hz)
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = true;
			}
			else if (bAnalog && gAddrIdx != 0)
			{
				TCCR0B = 3; // Normal mode, 1/64 prescal
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = false;
			}
			
			loadLongAddress(gAddrIdx);
			
			//Display Loc Address
			sprintf(gText, "%4d", gLocAddress);
			LCD.printBigNumber(gText, 5, 0);
			
			//Display LocProtcol
			if(gAddrIdx == 0)
			{
				LCD.printString("Ana", 14, 0);
				bupdate = false;
			}
			else if(!bupdate)
			{
				if(gLocProtcol == ADDR_DCC) {
					LCD.printString("DCC", 14, 0);
				} else {
					LCD.printString("MM ", 14, 0);
				}
				bupdate = true;
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			if( gLocAddress == 0) return;
			
			gLocFunc1 = (gLocFunc1 == 0) ? 1 : 0;
			
			if (gLocFunc1 == 1)
			{
				LCD.printString("On ", 7, 3);
			}
			else
			{
				LCD.printString("Off", 7, 3);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc1, gLocFunc1);
			
			if (gAddrIdx != gAddrIdxOld)
			{
				//Update LocAddress
				saveEEPROM(0);
				gAddrIdxOld = gAddrIdx;
				
				#ifdef _DEBUG
				Serial.print("LocNew:");
				Serial.println(gAddrIdx);
				#endif
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if( gLocAddress == 0) return;
			
			gLocFunc2 = (gLocFunc2 == 0) ? 1 : 0;
			if (gLocFunc2 == 1)
			{
				LCD.printString("On ", 14, 3);
			}
			else
			{
				LCD.printString("Off", 14, 3);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc2, gLocFunc2);
			
			if (gAddrIdx != gAddrIdxOld)
			{
				//Update LocAddress
				saveEEPROM(0);
				gAddrIdxOld = gAddrIdx;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment address
				gAddrIdx++;
				if(gAddrIdx >= MAX_LOCO) {
					gAddrIdx = 1;
				}
				
				loadLongAddress(gAddrIdx);
				
				//Display Loc Address
				sprintf(gText, "%4d", gLocAddress);
				LCD.printBigNumber(gText, 5, 0);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				if(gAddrIdx == 0)
				{
					gAddrIdx = MAX_LOCO - 1;
				}
				else
				{
					//Decrement address
					gAddrIdx--;
					if(gAddrIdx == 0) {
						gAddrIdx = MAX_LOCO - 1;
					}
				}
				
				loadLongAddress(gAddrIdx);
			
				//Display Loc No.
				sprintf(gText, "%4d", gLocAddress);
				LCD.printBigNumber(gText, 5, 0);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*******************************
    processModeFunction
********************************/

void processModeFn()
{
	if(gButton_val != gButton_old)
	{
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			//Increment func number
			if(gMode_Drive == DRV_FN1)
			{
				#ifdef _DEBUG
				digitalWrite(A3, HIGH); // 0ms~
				#endif
				
				gFunc1++;
				if(gFunc1 > MAX_FUNC) {
					gFunc1 = 0;
				}
				
				gPressCount = 0;
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc1);
				LCD.printBigNumber(gText, 5, 2);
				
				//Display Button
				LCD.printStringNOT("F", 11, 7);
				LCD.printStringNOT(gText, 12, 7);
				
				#ifdef _DEBUG
				digitalWrite(A3, LOW); // ~7.6ms
				#endif
			}
			else
			{
				gFunc2++;
				if(gFunc2 > MAX_FUNC) {
					gFunc2 = 0;
				}
				
				gPressCount = 0;
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc2);
				LCD.printBigNumber(gText, 12, 2);
				
				//Display Button
				LCD.printStringNOT("F", 15, 7);
				LCD.printStringNOT(gText,16, 7);
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gMode_Drive == DRV_FN1)
			{
				if(gFunc1 == 0) {
					gFunc1 = MAX_FUNC;
				} else {
					gFunc1--;
				}
				
				gPressCount = 0;
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc1);
				LCD.printBigNumber(gText, 5, 2);
				
				//Display Button
				LCD.printStringNOT("F", 11, 7);
				LCD.printStringNOT(gText, 12, 7);
			}
			else
			{
				if(gFunc2 == 0) {
					gFunc2 = MAX_FUNC;
				}
				else
				{
					gFunc2--;
				}
				
				gPressCount = 0;
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc2);
				LCD.printBigNumber(gText, 12, 2);
				
				//Display Button
				LCD.printStringNOT("F", 15, 7);
				LCD.printStringNOT(gText,16, 7);
			}
		}
		
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			if (gLocAddress == 0) return;
			
			gLocFunc1 = (gLocFunc1 == 0) ? 1 : 0;
			if (gLocFunc1 == 1)
			{
				LCD.printString("On ", 7, 3);
			}
			else
			{
				LCD.printString("Off", 7, 3);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc1, gLocFunc1);
			
			if (gFunc1 != gFunc1Old)
			{
				//Update Func1
				saveEEPROM(2);
				gFunc1Old = gFunc1;
			}
			
			if (gAddrIdx != gAddrIdxOld)
			{
				saveEEPROM(0);
				gAddrIdxOld = gAddrIdx;
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if (gLocAddress == 0) return;
			
			gLocFunc2 = (gLocFunc2 == 0) ? 1 : 0;
			if (gLocFunc2 == 1)
			{
				LCD.printString("On ", 14, 3);
			}
			else
			{
				LCD.printString("Off", 14, 3);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc2, gLocFunc2);
			
			if (gFunc2 != gFunc2Old)
			{
				//Update Func2
				saveEEPROM(3);
				gFunc2Old = gFunc2;
			}
			
			if (gAddrIdx != gAddrIdxOld)
			{
				saveEEPROM(0);
				gAddrIdxOld = gAddrIdx;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if (gPressCount > CONTINUOUS)
			{
				//Increment func number
				if(gMode_Drive == DRV_FN1)
				{
					gFunc1++;
					if(gFunc1 > MAX_FUNC) {
						gFunc1 = 0;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc1);
					LCD.printBigNumber(gText, 5, 2);
					
					//Display Button
					LCD.printStringNOT("F", 11, 7);
					LCD.printStringNOT(gText, 12, 7);
				}
				else
				{
					gFunc2++;
					if(gFunc2 > MAX_FUNC) {
						gFunc2 = 0;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc2);
					LCD.printBigNumber(gText, 12, 2);
					
					//Display Button
					LCD.printStringNOT("F", 15, 7);
					LCD.printStringNOT(gText,16, 7);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Decrement func number
				if(gMode_Drive == DRV_FN1)
				{
					if(gFunc1 == 0) {
						gFunc1 = MAX_FUNC;
					} else {
						gFunc1--;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc1);
					LCD.printBigNumber(gText, 5, 2);
					
					//Display Button
					LCD.printStringNOT("F", 11, 7);
					LCD.printStringNOT(gText, 12, 7);
				}
				else
				{
					if(gFunc2 == 0) {
						gFunc2 = MAX_FUNC;
					} else {
						gFunc2--;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc2);
					LCD.printBigNumber(gText, 12, 2);
					
					//Display Button
					LCD.printStringNOT("F", 15, 7);
					LCD.printStringNOT(gText,16, 7);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*******************************
    processModeAcc
********************************/

void processModeAcc()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			//Increment address
			gAccAddress++;
			
			if(gAccAddress > MAX_ACC) {
				gAccAddress = 1;
			}
			
			//Reset press count
			gPressCount = 0;
			
			//Display Acc No.
			sprintf(gText, "%2d", gAccAddress);
			LCD.printBigNumber(gText, 5, 4);
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			
			if(gAccAddress == 1) {
				gAccAddress = MAX_ACC;
			}
			else
			{
				//Decrement address
				gAccAddress--;
			}
			
			//Reset press count
			gPressCount = 0;
			
			//Display Acc No.
			sprintf(gText, "%2d", gAccAddress);
			LCD.printBigNumber(gText, 5, 4);
		}
		
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			if( gAccAddress == 0) return;
			
			//Diverse
			DSCore.SetTurnout(gAccProtcol + gAccAddress, 0);
			
			//Display Message
			LCD.printString("Div", 7, 5);
			LCD.printString("/", 10, 5);
			
			if(gAccAddress != gAccAddressOld)
			{
				//Update Acc
				saveEEPROM(4);
				gAccAddressOld = gAccAddress;
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if( gAccAddress == 0) return;
			
			//Straight
			DSCore.SetTurnout(gAccProtcol + gAccAddress, 1);
			
			//Display Message
			LCD.printString("Str", 7, 5);
			LCD.printString("|", 10, 5);
			
			if(gAccAddress != gAccAddressOld)
			{
				//Update Acc
				saveEEPROM(4);
				gAccAddressOld = gAccAddress;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment address
				gAccAddress++;
				
				if(gAccAddress > MAX_ACC) {
					gAccAddress = 1;
				}
				
				//Display Acc No.
				sprintf(gText, "%2d", gAccAddress);
				LCD.printBigNumber(gText, 5, 4);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				if(gAccAddress == 1) {
					gAccAddress = MAX_ACC;
				}
				else
				{
					//Decrement address
					gAccAddress--;
				}
				
				//Display Acc No.
				sprintf(gText, "%2d", gAccAddress);
				LCD.printBigNumber(gText, 5, 4);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

byte getFnIndex(byte inFunc, byte *inBit)
{
	byte aFnidx;
	
	if(inFunc < 8)
	{
		aFnidx = 0;
		*inBit = inFunc;
	}
	else if(inFunc >= 8 && inFunc < 16)
	{
		aFnidx = 1;
		*inBit = inFunc & 0x07;
	}
	else if(inFunc >= 16 && inFunc < 24)
	{
		aFnidx = 2;
		*inBit = inFunc & 0x0f;
	}
	else
	{
		aFnidx = 3;
		*inBit = inFunc & 0x07;
	}
	
	return aFnidx;
}

/*******************************
    getButtonStatus
********************************/

byte getButtonStatus()
{
	byte aButton_val = 0;
	
	//Input Buttons
	aButton_val = PIND & 0b11111100;
	aButton_val ^= 0x80;
	
	return ~aButton_val;
}

/*******************************
    checkButton
********************************/

bool checkButton(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( ((inBuf & inButtonBit) == 0) && ((inCurrent & inButtonBit) > 0))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*******************************
    checkButtonLeaved
********************************/

bool checkButtonLeaved(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( ((inBuf & inButtonBit) > 0) && ((inCurrent & inButtonBit) == 0))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*******************************
    checkButtonContinuous
********************************/

bool checkButtonContinuous(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( (inCurrent & inButtonBit) > 0)
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*******************************
    Load EEPROM
********************************/

byte loadEEPROM()
{
	byte aAdrIxd = 3;
	
	if( EEPROM.read(0) == 0xCC)
	{
		aAdrIxd = EEPROM.read(2);
		if(aAdrIxd >= MAX_LOCO) aAdrIxd = MAX_LOCO - 1;
		
		eepProtcol = EEPROM.read(4);
		if( eepProtcol > 1)
		{
			eepProtcol = 0;
		}
		
		gFunc1 = EEPROM.read(5);
		if(gFunc1 > MAX_FUNC) gFunc1 = 0;
		gFunc2 = EEPROM.read(6);
		if(gFunc2 > MAX_FUNC) gFunc2 = 1;
		
		gAccAddress = EEPROM.read(7);
		if (gAccAddress > MAX_ACC)
		{
			gAccAddress = MAX_ACC;
		}
		
		#ifdef _DEBUG
		Serial.println("");
		Serial.println("loadEDEPROM()");
		Serial.print("AddrIdx=");
		Serial.print(aAdrIxd);
		
		Serial.print(",Protocol=");
		Serial.println(eepProtcol);
		
		Serial.print("Func1=");
		Serial.print(gFunc1);
		Serial.print(", Func2=");
		Serial.println(gFunc2);
		
		Serial.print("Acc=");
		Serial.println(gAccAddress);
		Serial.println("");
		#endif
		
	}
	
	return aAdrIxd;
}

void loadLongAddress(byte inAdrIdx)
{
	byte eepaddr = inAdrIdx * 2 + EEPOFFSET;
	
	if(inAdrIdx == 0)
	{
		gLocAddress = 0;
		return;
	}
	
	if( EEPROM.read(0) == 0xCC)
	{
		// about 10usec
		gLocAddress = (EEPROM.read(eepaddr + 1) << 8) + EEPROM.read(eepaddr);
		if(gLocAddress > 9999)
		{
			gLocAddress = 9999;
		}
	}
	else
	{
		#ifdef _DEBUG
		Serial.print("Init eepAddress...");
		#endif
		
		//Mint can ID
		EEPROM.write(0, 0xCC);
		
		for(int i=1; i<MAX_LOCO; i++)
		{
			gLocAddress = i;
			saveLongAddress(i);
		}
		
		gLocAddress = (EEPROM.read(eepaddr + 1) << 8) + EEPROM.read(eepaddr);
		
		#ifdef _DEBUG
		Serial.println("");
		Serial.println("Done.");
		#endif
	}
}

void saveLongAddress(byte inAdrIdx)
{
	byte eepaddr = inAdrIdx * 2 + EEPOFFSET;
	
	#ifdef _DEBUG
	digitalWrite(A3, HIGH);
	#endif
	
	EEPROM.write(eepaddr, lowByte(gLocAddress));
	EEPROM.write(eepaddr + 1, highByte(gLocAddress));
	EEPROM.write(2, inAdrIdx);
	
	#ifdef _DEBUG
	digitalWrite(A3, LOW);
	Serial.print("SaveLAddr:");
	Serial.print(gLocAddress);
	Serial.print("[");
	Serial.print(inAdrIdx);
	Serial.println("]");
	#endif
}

/*******************************
    Save EEPROM
********************************/

void saveEEPROM(byte inMode)
{
	if( EEPROM.read(0) == 0xCC)
	{
		switch(inMode)
		{
			case 0:
				EEPROM.write(2, gAddrIdx);
				break;
				
			case 1:
				EEPROM.write(4, eepProtcol);
				break;
				
			case 2:
				EEPROM.write(5, gFunc1);
				break;
				
			case 3:
				EEPROM.write(6, gFunc2);
				break;
				
			case 4:
				EEPROM.write(7, gAccAddress);
				break;
				
		}
		
		#ifdef _DEBUG
		Serial.print("saveEEPROM:");
		Serial.println(inMode);
		#endif
	}
	else
	{
		//Mint can ID
		EEPROM.write(0, 0xCC);
		
		//Data write
		EEPROM.write(1, 0);
		EEPROM.write(2, gAddrIdx);
		EEPROM.write(4, eepProtcol);
		
		EEPROM.write(5, gFunc1);
		EEPROM.write(6, gFunc2);
		
		EEPROM.write(7, gAccAddress);
	}
}
