/*********************************************************************
 * MintCan2 DCC/MM Controler r0.1
 *
 * Copyright (C) 2018 twaydcc
 *
 */

#include <arduino.h>
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <EEPROM.h>
#include <SPI.h>

#include "DSCoreM_Type.h"
#include "DSCoreM_Common.h"
#include "DSCoreM_List.h"
#include "DSCoreM_DCC.h"
#include "DSCoreM_MM2.h"
#include "DSCoreM.h"
#include "OzOLED_spi.h"

// Definition constance

//#define _DEBUG
#define _DEBUGPORT
#define _SPEEDBAR

#define BUTTON_SEL  0b00000100
#define BUTTON_UP   0b00001000
#define BUTTON_DOWN 0b00010000
#define BUTTON_F1   0b00100000
#define BUTTON_F2   0b01000000
#define BUTTON_DIR  0b10000000

#define DCC_MODE	0
#define MM_MODE		1

#define MODE_LADR	1
#define MODE_CVADR	2
#define MODE_CVWR	3
#define MODE_ROUTE	4
#define MODE_CV		5
#define MODE_LOC	6

#define DRV_LOC		0
#define DRV_ACC		1
#define DRV_ROUTE	2

#define MAX_LOCO	51
#define MAX_FUNC	28
#define MAX_ACC		2044
#define MAX_MMACC	320
#define MAX_ROUTE	25
#define MAX_RTADR	10

#define MAX_CV		99

#define DC_PIN		16
#define RST_PIN		15
#define CS_PIN		8

#define CONTINUOUS	4
#define EEPOFFSET	12
#define EEPROUTE	112

// Class
DSCoreLib DSCore;
OzOLED LCD(DC_PIN, RST_PIN, CS_PIN);
char gText[8];

//Loc mode
byte gAddrIdx = 3;     // LocAddress index
byte gAddrIdxOld;
word gLocAddress = 3;
boolean bAnalog = false;

byte gCol = 0;
byte gColValue[4];

//byte gConsAddress = 6;
//byte gEStop = 0;

byte gMode_Main = MODE_LOC;
byte gMode_Drive = DRV_LOC;
byte gUpDwnPos = 0;
byte gUpDwnPosOld = 0;
byte gPartialCount = 0;
boolean gDualButton = false;

//Protcol
byte eepProtcol = DCC_MODE;
word gLocProtcol = ADDR_DCC;
word gAccProtcol = ADDR_ACC_DCC;

//Func
byte gFunc1 = 0;
byte gFunc1Old;
byte gFunc2 = 1;
byte gFunc2Old;
byte gLocFunc1 = 0;
byte gLocFunc2 = 0;

//Acc
word gAccAddr = 1;
word gAccAddrOld;
byte gRouteNo = 1;
byte gRouteNoOld;
byte gAccList = 1;
byte gAccSwitch;
byte gRouteCount = 0;
byte gRoutePos;

//CV
byte gCVAddress = 1;
byte gCVValue = 0;
byte gCVWrCount = 0;

byte gButton_val = 0;
byte gButton_old = 0;

byte gPressCount = 0;
unsigned long gTime_100ms = 0;

// Functions
void displayInit();
void displayModeLongAddr();
void displayModeCV29();
void displayModeCVAddr();
void displayModeCV();
void displayModeLoc();
void displayModeFunc();
void displayModeAcc();
void displayModeRoute();
void displayDiffer();
void displayStringsList(byte inX, byte inY, word inAddr, boolean inBig);

void processModeLongAddr();
void processModeCVAddr();
void processModeWrAddr();
void processModeCV();
void processModeDial();
void processModeSel();
void processModeLoc();
void processModeFunc();
void processModeAcc();
void processModeRoute();
void processSendRoute();

void mkColValArray(byte *array, word inData);
byte getFnIndex(byte inFunc, byte *inBit);

byte getButtonStatus();
bool checkButton(byte inBuf, byte inCurrent, byte inButtonBit);
bool checkButtonLeaved(byte inBuf, byte inCurrent, byte inButtonBit);
bool checkButtonContinuous(byte inBuf, byte inCurrent, byte inButtonBit);

void loadEEPROM();
void saveEEPROM(byte inItem);
void loadLongAddrEEP(byte inIdx); //load gLocAddress
void saveLongAddrEEP(byte inIdx, word inAddr);
word loadRouteAddrEEP(byte inRout, byte inAccList);
void saveRouteAddrEEP(byte inRout, byte inAccList, word inAccAddr);

/*******************************
    Setup
********************************/

void setup()
{
	pinMode(10, OUTPUT);
	digitalWrite(10, LOW);
	pinMode(9, OUTPUT);
	digitalWrite(9, LOW);
	
	//Serial
	#ifdef _DEBUG
	Serial.begin(115200);
	while (!Serial);

	Serial.println("-----------------------");
	Serial.println("twaydcc MintCanIV r0.1 ");
	Serial.println("-----------------------");
	Serial.println("100 Ready");
	#endif
	
	#ifdef _DEBUGPORT
	pinMode(A3, OUTPUT);
	digitalWrite(A3, HIGH);
	#endif
	
	//OLDE init
	displayInit();
	
	//Analog input
	pinMode(A0, INPUT);
	digitalWrite(A0, LOW); // disable internal pullup
	analogReference(DEFAULT); // set analog reference 5V(VCC) 
	
	//Faster Analog read
	ADCSRA = ADCSRA & 0xf8;
	ADCSRA = ADCSRA | 0x04;
	
	//Function button
	pinMode(2, INPUT);
	pinMode(3, INPUT);
	pinMode(4, INPUT);
	pinMode(5, INPUT);
	pinMode(6, INPUT);
	//Direction button
	pinMode(7, INPUT);
	// input pullup
	digitalWrite(2, HIGH);
	digitalWrite(3, HIGH);
	digitalWrite(4, HIGH);
	digitalWrite(5, HIGH);
	digitalWrite(6, HIGH);
	digitalWrite(7, HIGH);
	
	//Load EEPROM
	loadEEPROM();
	gAddrIdxOld = gAddrIdx;
	loadLongAddrEEP(gAddrIdx);
	
	gFunc1Old = gFunc1;
	gFunc2Old = gFunc2;
	
	gAccAddrOld = gAccAddr;
	gRouteNoOld = gRouteNo;
	
	//Set default button status
	gButton_val = getButtonStatus();
	
	//mode check
	if((gButton_val & BUTTON_F2) > 0) {
		gMode_Main = MODE_LADR;
		
	} else if((gButton_val & BUTTON_F1) > 0) {
		gMode_Main = MODE_CV;
	} else if((gButton_val & BUTTON_UP) > 0) {
		// Change to DCC mode
		eepProtcol = DCC_MODE;
		//Update Protocol
		saveEEPROM(3);
		
	} else if((gButton_val & BUTTON_DOWN) > 0) {
		// Change to MM2 mode
		eepProtcol = MM_MODE;
		//Update Protocol
		saveEEPROM(3);
	}
	gButton_old = gButton_val;
	
	/* Protcol changer */
	if( eepProtcol == MM_MODE)
	{
		/* MM2 Mode */
		gLocProtcol = ADDR_MM2;
		gAccProtcol = ADDR_ACC_MM2;
	}
	else
	{
		/* DCC Mode */
		gLocProtcol = ADDR_DCC;
		gAccProtcol = ADDR_ACC_DCC -1;
	}
	
	//Init DSCore
	DSCore.Init();
	
	//LCD display
	switch( gMode_Main) {
		case MODE_LADR:
			displayModeLongAddr();
			break;
		case MODE_CV:
			displayModeCV();
			break;
		case MODE_LOC:
			displayModeLoc();
			break;
	}
	
	while ((gButton_val & BUTTON_UP) > 0 || (gButton_val & BUTTON_DOWN) > 0)
	{
		gButton_val = getButtonStatus();
		
		//Reset press count
		gPressCount = 0;
	}
	
	gTime_100ms = millis();
	
	#ifdef _DEBUG
	Serial.println("Done");
	#endif
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

/*******************************
    Main loop
********************************/

void loop()
{
	//Scan
	if( (gLocAddress != 0) && (gMode_Main == MODE_LOC))
	{
		//Power on
		if( DSCore.IsPower() == false)
		{
			DSCore.SetPower(true);
		}
		
		DSCore.Scan();
	}
	
	// 100ms interval
	if( (millis() - gTime_100ms) >= 100)
	{
		gButton_val = getButtonStatus();
		
		switch (gMode_Main) {
		  case MODE_LADR:
			processModeLongAddr();
			break;
		  case MODE_CVADR:
			processModeCV29();
			break;
		  case MODE_CVWR:
			processModeWrAddr();
			break;
			
		  case MODE_CV:
			processModeCV();
			break;
			
		  case MODE_LOC:
			//Update dial for speed control
			processModeDial();
			
			if( gLocAddress > 0) processModeSel();
			
			if(gUpDwnPos == 0) processModeLoc();
			
			if(gMode_Drive == DRV_LOC) {
				displayModeFunc();
				processModeFunc();
			}
			else if(gMode_Drive == DRV_ACC) {
				displayModeAcc();
				processModeAcc();
			}
			else if(gMode_Drive == DRV_ROUTE) {
				displayModeRoute();
				processModeRoute();
				if(gRouteCount > 0) {
					processSendRoute();
				}
			}
			break;
		}
		
		if(gButton_val != gButton_old)
		{
			gButton_old = gButton_val;
		}
		
		//Reset task
		gTime_100ms = millis();
	}
}

/*******************************
    OLED init
********************************/

void displayInit()
{
	LCD.init();
	/*
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	LCD.clearDisplay();           //clear the screen and set start position to top left corner
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
	LCD.setNormalDisplay();       //Set display to Normal mode
	LCD.setPageMode();            //Set addressing mode to Page Mode
	*/
}

/*******************************
    DisplayMode
********************************/

/*** Display LongAddress ***/
void displayModeLongAddr()
{
	LCD.clearDisplay();
	
	LCD.printString("<Loco Addr asign>", 1, 0);
	// Src address
	LCD.printString("Src", 0, 2);
	LCD.drawIcon(5, 2, 9, false);
	LCD.drawIcon(5, 3, 10, false);
	sprintf(gText, "%2d", gAddrIdx);
	LCD.printBigNumber(gText, 6, 2);
	
	// Distnation address
	LCD.printString("Dist", 0, 4);
	displayStringsList(6, 4, gLocAddress, true);
	
	//Button
	LCD.printStringNOT("SEL", 0, 7);
}

/*** Display CV29 ***/
void displayModeCV29()
{
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	LCD.clearDisplay();
	
	LCD.printString("<Loco Addr write>", 1, 0);
	//CV29
	LCD.printString("CV29", 0, 2);
	LCD.drawIcon(5, 2, 9, false);
	LCD.drawIcon(5, 3, 10, false);
	//CV29 bits
	if(gLocAddress < 100) {
		LCD.printBigNumber("000010", 5, 2);
		// val
		LCD.printBigNumber("02", 15, 2);
		//Address
		LCD.printString("CV1", 0, 5);
		sprintf(gText, "%2d", gLocAddress);
		LCD.printString(gText, 5, 5);
	} else {
		LCD.printBigNumber("100010", 5, 2);
		LCD.printBigNumber("34", 15, 2);
		//Address
		LCD.printString("CV17", 0, 5);
		sprintf(gText, "%2d", 192 + gLocAddress / 255);
		LCD.printString(gText, 5, 5);
		LCD.printString("CV18", 10, 5);
		sprintf(gText, "%2d", gLocAddress % 255);
		LCD.printString(gText, 15, 5);
	}
	
	// select bit
	LCD.printString("*", 6, 4);
	
	//Button
	LCD.printStringNOT("SEL", 0, 7);
	LCD.printStringNOT("S+", 5, 7);
	LCD.drawIcon(7, 7, 10, true);
	LCD.printStringNOT("CVW", 10, 7);
	LCD.printStringNOT("ESC", 15, 7);
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

/*** Display CVAddress ***/
void displayModeCVAddr()
{
	//CV29 bits
	for(int i=2; i<8; i++) {
		if(bitRead(gCVValue, 7 - i) == 0) {
			LCD.printString("0", 3 + i, 2);
		} else {
			LCD.printString("1", 3 + i, 2);
		}
	}
	//CV29 val
	sprintf(gText, "%2d", gCVValue);
	LCD.printString(gText, 15, 2);
	LCD.printString("  ", 15, 3);
	// clear CV29 bits selecter
	LCD.printString("       ", 5, 3);
	
	//CV1 or CV17-18
	if(gLocAddress < 100) {
		//Address
		sprintf(gText, "%2d", gLocAddress);
		LCD.printBigNumber(gText, 5, 4);
	} else {
		sprintf(gText, "%2d", 192 + gLocAddress / 255);
		LCD.printBigNumber(gText, 5, 4);
		sprintf(gText, "%2d", gLocAddress % 255);
		LCD.printBigNumber(gText, 15, 4);
	}
	
	//Button
	LCD.printString("   ", 5, 7);
}

/*** Display CV ***/
void displayModeCV()
{
	LCD.printString("<CV programmer>", 2, 0);
	//CV No.
	LCD.printString("CV ", 1, 2);
	LCD.drawIcon(5, 2, 9, false);
	LCD.drawIcon(5, 3, 10, false);
	sprintf(gText, "%2d", 1);
	LCD.printBigNumber(gText, 7, 2);
	//CV value
	LCD.printString("Val", 1, 4);
	LCD.printBigNumber("003", 6, 4);
	gColValue[2] = 3;
	//Button
	LCD.printStringNOT("SEL", 0, 7);
}


/*** Display Loco ***/
void displayModeLoc()
{
	// Loco
	LCD.printString("Loco", 0, 0);
	// Protcol
	if(gLocAddress == 0) {
		LCD.printString("Ana", 14, 0);
	} else {
		if(gLocProtcol == ADDR_DCC) {
			LCD.printString("DCC", 14, 0);
		} else {
			LCD.printString("MM ", 14, 0);
		}
	}
	
	// Loco cursol
	LCD.drawIcon(5, 0, 9, false);
	LCD.drawIcon(5, 1, 10, false);
	// Loco address
	sprintf(gText, "%4d", gLocAddress);
	LCD.printBigNumber(gText, 6, 0);

	// display speed
	#ifdef _SPEEDBAR
	LCD.drawSpeedBar( 14, 1, 0, false, false);
	#else
	sprintf(gText, "%3d", 0);
	LCD.printString(gText, 14, 1);
	#endif
	
	// Direction
	LCD.printString(" ", 13, 1);
	LCD.drawIcon(17, 1, 11, false); // Fwd
	
	gPartialCount = 1;
	displayModeFunc();
	gPartialCount = 2;
	displayModeFunc();
	
	//Display Button
	LCD.printStringNOT("SEL", 0, 7);
}

/*** Display Function ***/
void displayModeFunc()
{
	if(gPartialCount == 0) return;
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	if(gPartialCount == 1)
	{
		// Fn number
		LCD.printString("Func", 0, 3);
		sprintf(gText, "%2d", gFunc1);
		LCD.printBigNumber(gText, 6, 3);
		// clear Acc addr
		LCD.printString("  ", 8, 3);
		LCD.printString("  ", 8, 4);
		// clear Acc '****'
		if(gLocFunc1 == 1) {
			LCD.printString("On  ", 6, 5);
		} else {
			LCD.printString("Off ", 6, 5);
		}
		
		//Button
		LCD.printStringNOT("S", 5, 7);
		LCD.drawIcon(6, 7, 9, true);
		LCD.drawIcon(7, 7, 10, true);
		
		gPartialCount = 2;
	}
	else if(gPartialCount == 2)
	{
		// clear Route AccList
		LCD.printString("  ", 10, 3);
		// Fx number
		sprintf(gText, "%2d", gFunc2);
		LCD.printBigNumber(gText, 12, 3);
		// clear Acc stat or Route addr
		LCD.printString("    ", 14, 3);
		LCD.printString("    ", 14, 4);
		
		if(gUpDwnPos == 2) {
			// clear Fn cursol
			LCD.printString(" ", 5, 3);
			LCD.printString(" ", 5, 4);
			// Fx cursol
			LCD.drawIcon(11, 3, 9, false);
			LCD.drawIcon(11, 4, 10, false);
		}
		
		if(gLocFunc2 == 1) {
			LCD.printString("On  ", 12, 5);
		} else {
			LCD.printString("Off ", 12, 5);
		}
		
		//Button
		LCD.printStringNOT("F", 10, 7);
		sprintf(gText, "%2d", gFunc1);
		LCD.printStringNOT(gText, 11, 7);
		LCD.printStringNOT("F", 15, 7);
		sprintf(gText, "%2d", gFunc2);
		LCD.printStringNOT(gText,16, 7);
		
		gPartialCount = 0;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

/*** Display Acc ***/
void displayModeAcc()
{
	if(gPartialCount == 0) return;
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	if(gPartialCount == 1)
	{
		// Acc
		LCD.printString("Acc ", 0, 3);
		if(gUpDwnPos == 0) {
			LCD.printString("   ", 6, 5);
		} else {
			LCD.printString("*   ", 6, 5);
		}
		
		// Acc cursol
		if(gUpDwnPosOld == 2) {
			LCD.drawIcon(5, 3, 9, false);
			LCD.drawIcon(5, 4, 10, false);
		}
		displayStringsList(6, 3, gAccAddr, true);
		
		gPartialCount = 2;
	}
	else
	{
		// claer Fx
		LCD.printString("    ", 10, 3);
		LCD.printString("    ", 10, 4);
		LCD.printString("   ", 12, 5);
		
		LCD.printString("Div/", 14, 4);
		
		//Button
		LCD.printStringNOT("DIV", 10, 7);
		LCD.printStringNOT("STR", 15, 7);
		
		gPartialCount = 0;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

/*** Display Route ***/
void displayModeRoute()
{
	if(gPartialCount == 0) return;
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	if(gPartialCount == 1)
	{
		// Route
		LCD.printString("Rout", 0, 3);
		
		if(gUpDwnPos == 1) {
			LCD.printString("*", 6, 5);
		}
		
		sprintf(gText, "%2d", gRouteNo);
		LCD.printBigNumber(gText, 6, 3);
		// RouteNo cursol
		if(gUpDwnPosOld == 2) {
			LCD.drawIcon(5, 3, 9, false);
			LCD.drawIcon(5, 4, 10, false);
		}
		// clear Fn
		LCD.printString("   ", 6, 5);
		// clear Acc 00
		LCD.printString(" ", 8, 3);
		LCD.printString("     ", 8, 4);
		// clear Acc stat
		LCD.printString("    ", 14, 4);
		// Acc stat
		LCD.printString("Div/", 12, 5);
		
		gPartialCount = 2;
	}
	else
	{
		LCD.printString("adr", 9, 3);
		sprintf(gText, "%1d", gAccList);
		LCD.printString(gText, 12, 3);
		// clear Fx
		LCD.printString(" ", 13, 3);
		LCD.printString(" ", 13, 4);
		// AccList addr
		gAccAddr = loadRouteAddrEEP(gRouteNo, gAccList);
		displayStringsList(14, 3, gAccAddr, false);
		
		//Button
		LCD.printStringNOT("DIV", 10, 7);
		LCD.printStringNOT("STR", 15, 7);
		
		gPartialCount = 0;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

void displayDiffer()
{
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	switch(gUpDwnPos) {
	  case 0:
		// Loco cursol
		LCD.drawIcon(5, 0, 9, false);
		LCD.drawIcon(5, 1, 10, false);
		// clear Fn cursol
		LCD.printString(" ", 5, 3);
		LCD.printString(" ", 5, 4);
		// clear Fx cursol
		if(gMode_Drive != DRV_ROUTE)
		{
			LCD.printString(" ", 11, 3);
			LCD.printString(" ", 11, 4);
			
			if(gMode_Drive == DRV_ACC) {
				LCD.printString(" ", 9, 5);
			}
		}
		
		if(gMode_Drive == DRV_ROUTE)
		{
			// clear '*'
			LCD.printString("      ", 12, 4);
			LCD.printString(" ", 9, 5);
			
			// Acc stat
			if(gAccSwitch == 0) {
				LCD.printString("Div/", 12, 5);
			} else {
				LCD.printString("Str|", 12, 5);
			}
			//Button
			LCD.printStringNOT("DIV", 10, 7);
			LCD.printStringNOT("STR", 15, 7);
		}
		break;
		
	  case 1:
		// clear Loco cursol
		LCD.printString(" ", 5, 0);
		LCD.printString(" ", 5, 1);
		// Fn cursol
		LCD.drawIcon(5, 3, 9, false);
		LCD.drawIcon(5, 4, 10, false);
		
		if(gMode_Drive == DRV_LOC)
		{
			// clear Acc stat or Route adr
			LCD.printString("    ", 14, 3);
		}
		else if((gMode_Drive == DRV_ACC) && (gCol==0))
		{
			// point top of Acc addr
			LCD.printString("*", 6, 5);
		}
		else if(gMode_Drive == DRV_ROUTE)
		{
			// clear Edit AccList '*'
			LCD.printString(" ", 12, 4);
			// clear Edit addr point
			LCD.printString("    ", 14, 4);
			// Acc stat
			if(gAccSwitch == 0) {
				LCD.printString("Div/", 12, 5);
			} else {
				LCD.printString("Str|", 12, 5);
			}
			//Button
			LCD.printStringNOT("DIV", 10, 7);
			LCD.printStringNOT("STR", 15, 7);
		}
		break;
		
	  case 2:
		// clear Fn cursol
		LCD.printString(" ", 5, 3);
		LCD.printString(" ", 5, 4);
		
		if(gMode_Drive == DRV_LOC)
		{
			// Fx cursol
			LCD.drawIcon(11, 3, 9, false);
			LCD.drawIcon(11, 4, 10, false);
		}
		else if(gMode_Drive == DRV_ROUTE)
		{
			// clear Loco cursol
			LCD.printString(" ", 5, 0);
			LCD.printString(" ", 5, 1);
			// Edit AccList point
			LCD.printString("*", 12, 4);
			// clear '*'
			LCD.printString("    ", 14, 4);
			// clear Div/Str
			LCD.printString("    ", 12, 5);
			//Button
			LCD.printStringNOT("ENT", 10, 7);
			LCD.printStringNOT("ESC", 15, 7);
		}
		break;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

void displayStringsList(byte inX, byte inY, word inAddr, boolean inBig)
{
	mkColValArray(gColValue, inAddr);
	if(inBig)
	{
		LCD.printBigNumber(&gText[0], inX, inY);
		LCD.printBigNumber(&gText[2], inX+1, inY);
		LCD.printBigNumber(&gText[4], inX+2, inY);
		LCD.printBigNumber(&gText[6], inX+3, inY);
	}
	else
	{
		LCD.printString(&gText[0], inX, inY);
		LCD.printString(&gText[2], inX+1, inY);
		LCD.printString(&gText[4], inX+2, inY);
		LCD.printString(&gText[6], inX+3, inY);
	}
}

/*******************************
    processMode
********************************/

/*** Process LongAddress ***/
void processModeLongAddr()
{
	static word aDstAddr=0;
	
	if(gButton_val != gButton_old)
	{
		// select Spurce address or Distnation address
		if(checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			//モード切替
			if(gUpDwnPos == 0)
			{
				gUpDwnPos = 1;
				// clear Src cursol
				LCD.printString(" ", 5, 2);
				LCD.printString(" ", 5, 3);
				// Dist cursol
				LCD.drawIcon(5, 4, 9, false);
				LCD.drawIcon(5, 5, 10, false);
				// select Col0
				LCD.printString("*", 6, 6);
				//Button
				LCD.printStringNOT("CVW", 10, 7);
				LCD.printStringNOT("ENT", 15, 7);
			}
			else
			{
				gCol++;
				if(gCol > 3) {
					gCol = 0;
					gUpDwnPos = 0;
					// clear Dist cursol
					LCD.printString(" ", 5, 4);
					LCD.printString(" ", 5, 5);
					// Src cursol
					LCD.drawIcon(5, 2, 9, false);
					LCD.drawIcon(5, 3, 10, false);
					// clear Col3 selecter
					LCD.printString(" ", 9, 6);
					
					//clear ENT Button
					LCD.printString("   ", 10, 7);
					LCD.printString("   ", 15, 7);
				}
				else
				{
					// clear prev Col
					LCD.printString(" ", 5 + gCol, 6);
					// next Col
					LCD.printString("*", 6 + gCol, 6);
				}
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			//Increment address
			if(gUpDwnPos == 0)
			{
				//Src select
				gAddrIdx++;
				if(gAddrIdx >= MAX_LOCO) {
					gAddrIdx = 1;
				}
				loadLongAddrEEP(gAddrIdx);
				
				//Display Source address
				sprintf(gText, "%2d", gAddrIdx);
				LCD.printBigNumber(gText, 6, 2);
				
				//Display dist address
				displayStringsList(6, 4, gLocAddress, true);
			}
			else
			{
				//Dist select
				gColValue[gCol]++;
				if(gColValue[gCol] > 9) gColValue[gCol] = 0;
				
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 4);
			}
			
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			//Decrement address
			if(gUpDwnPos == 0)
			{
				//Src select
				if(gAddrIdx == 1) {
					gAddrIdx = MAX_LOCO -1;
				} else {
					gAddrIdx--;
				}
				loadLongAddrEEP(gAddrIdx);
				
				//Display Source address
				sprintf(gText, "%2d", gAddrIdx);
				LCD.printBigNumber(gText, 6, 2);
				
				//Display dist address
				displayStringsList(6, 4, gLocAddress, true);
			}
			else
			{
				//Dist select
				if(gColValue[gCol] == 0) {
					gColValue[gCol] = 9;
				} else {
					gColValue[gCol]--;
				}
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 4);
			}
			
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			// CVW
			if(gUpDwnPos == 1) {
				gMode_Main = MODE_CVADR;
				gCVWrCount = 0;
				if(gLocAddress < 100) {
					gCVValue = 2;
				} else {
					gCVValue = 34;
				}
				gCol = 3;
				displayModeCV29();
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			// Enter long address
			if(gUpDwnPos == 1) {
				LCD.printString("****", 6, 6);
				
				gLocAddress = (gColValue[0] * 1000 + gColValue[1] * 100
								+ gColValue[2] * 10 + gColValue[3]);
				saveLongAddrEEP(gAddrIdx, gLocAddress);
				
				gUpDwnPos = 0;
				gCol = 0;
				// clear Dist cursol
				LCD.printString(" ", 5, 4);
				LCD.printString(" ", 5, 5);
				// Src cursol
				LCD.drawIcon(5, 2, 9, false);
				LCD.drawIcon(5, 3, 10, false);
				
				// clear Dist pos selecter
				_delay_ms(500);
				LCD.printString("    ", 6, 6);
				//Button
				LCD.printString("   ", 10, 7);
				LCD.printString("   ", 15, 7);
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				if(gUpDwnPos == 1) {
					gColValue[gCol]++;
					if(gColValue[gCol] > 9) gColValue[gCol] = 0;
					
					sprintf(gText, "%1d", gColValue[gCol]);
					LCD.printBigNumber(gText, 7 + gCol, 4);
				}
			}
			
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			//Decrement address
			if(gUpDwnPos == 1)
			{
				if(gColValue[gCol] == 0) {
					gColValue[gCol] = 9;
				} else {
					gColValue[gCol]--;
				}
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 7 + gCol, 4);
			}
			
			gPressCount = 0;
		}
	}
}

/*** Process CVAddress ***/
void processModeCV29()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
			{
				// skip CV29
				gCVWrCount = 1;
				displayModeCVAddr();
			}
		}
		else if(checkButtonLeaved(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(gCVWrCount == 0) {	//CV29
				gCol++;
				if(gCol == 6) {
					gCol++;
					LCD.printString(" ", 8, 4);
				}
				
				if(gCol > 7) {
					LCD.printString(" ", 10, 4);
					LCD.printString("*", 6, 4);
					gCol = 3;
				} else {
					LCD.printString(" ", 2 + gCol, 4);
					LCD.printString("*", 3 + gCol, 4);
				}
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			if(gCVWrCount == 0) {	//CV29
				if(bitRead(gCVValue, 7 - gCol) == 0) {
					bitWrite(gCVValue, 7 - gCol, 1);
					LCD.printBigNumber("1", 3 + gCol, 2);
				} else {
					bitWrite(gCVValue, 7 - gCol, 0);
					LCD.printBigNumber("0", 3 + gCol, 2);
				}
				sprintf(gText, "%2d", gCVValue);
				LCD.printBigNumber(gText, 15, 2);
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gCVWrCount == 0) {	//CV29
				if(bitRead(gCVValue, 7 - gCol) == 0) {
					bitWrite(gCVValue, 7 - gCol, 1);
					LCD.printBigNumber("1", 3 + gCol, 2);
				} else {
					bitWrite(gCVValue, 7 - gCol, 0);
					LCD.printBigNumber("0", 3 + gCol, 2);
				}
				sprintf(gText, "%2d", gCVValue);
				LCD.printBigNumber(gText, 15, 2);
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			gMode_Main = MODE_CVWR;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			gMode_Main = MODE_LADR;
			gUpDwnPos = 0;
			gCol = 0;
			displayModeLongAddr();
		}
	}
}

/*** Process Write Address ***/
void processModeWrAddr()
{
	static byte aInterval = 0;
	
	aInterval++;
	//if(aInterval > 5)
	{
		aInterval = 0;
		
		switch (gCVWrCount) {
		  case 0: // CV29
			LCD.drawIcon(10, 6, 2, false);
			DSCore.WriteConfig_Dir(29, gCVValue);
			gCVWrCount = 1;
			break;
			
		  case 1:
			if(gLocAddress < 100) { // CV1
				LCD.drawIcon(12, 6, 2, false);
				gCVValue = gLocAddress;
				DSCore.WriteConfig_Dir(1, gCVValue);
				gCVWrCount = 3;
			} else { // CV17
				LCD.drawIcon(11, 6, 2, false);
				gCVValue = 192 + gLocAddress / 256;
				DSCore.WriteConfig_Dir(17, gCVValue);
				gCVWrCount = 2;
			}
			break;
			
		  case 2: // CV18
			LCD.drawIcon(12, 6, 2, false);
			gCVValue = gLocAddress % 256;
			DSCore.WriteConfig_Dir(18, gCVValue);
			
			gCVWrCount = 3;
			break;
			
		  case 3: // Exit
			LCD.printString("   ", 13, 6);
			gMode_Main = MODE_LADR;
			gUpDwnPos = 0;
			displayModeLongAddr();
			break;
		}
		
		#ifdef _DEBUG
		Serial.print("CV write:");
		Serial.print(gCVWrCount);
		Serial.print(",");
		Serial.println(gCVValue);
		#endif
	}
}

/*** Process CV programmer ***/
void processModeCV()
{
	if(gButton_val != gButton_old)
	{
		// select CVAddress, CVValue
		if(checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(gUpDwnPos == 0) {
				gUpDwnPos = 1;
				// clear Value cursol
				LCD.printString(" ", 5, 2);
				LCD.printString(" ", 5, 3);
				// CV cursol
				LCD.drawIcon(5, 4, 9, false);
				LCD.drawIcon(5, 5, 10, false);
				// select Col0
				LCD.printString("*", 6, 6);
				
				//Button
				LCD.printStringNOT("CVW", 10, 7);
			} else {
				gCol++;
				if(gCol > 2) {
					gCol = 0;
					gUpDwnPos = 0;
					// clear Value cursol
					LCD.printString(" ", 5, 4);
					LCD.printString(" ", 5, 5);
					// CV cursol
					LCD.drawIcon(5, 2, 9, false);
					LCD.drawIcon(5, 3, 10, false);
					// Col selecter
					LCD.printString(" ", 8, 6);
					//clear CVW Button
					LCD.printString("   ", 10, 7);
				}
				else
				{
					// clear prev Col
					LCD.printString(" ", 5 + gCol, 6);
					// next Col
					LCD.printString("*", 6 + gCol, 6);
				}
			}
		}
		// up/down CVAddress or CVValue
		else if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			switch(gUpDwnPos) {
			case 0: //CV No select
				//Increment CV no
				gCVAddress++;
				if(gCVAddress > MAX_CV) gCVAddress = 1;
				
				//Display CV No.
				sprintf(gText, "%2d", gCVAddress);
				LCD.printBigNumber(gText, 7, 2);
				break;
				
			case 1: //CV Value select
				//Increment CV value
				gColValue[gCol]++;
				if(gCol == 0) {
					if(gColValue[gCol] > 2) gColValue[0] = 0;
				}
				else {
					if(gColValue[0] == 2) {
						if(gColValue[gCol] > 5) gColValue[gCol] = 0;
					} else {
						if(gColValue[gCol] > 9) gColValue[gCol] = 0;
					}
				}
				
				//Display CV Val.
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 4);
				break;
			}
			
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			switch(gUpDwnPos) {
			case 0: //CV No select
				//Decrement CV no
				if(gCVAddress == 1) {
					gCVAddress = MAX_CV;
				} else {
					//Decrement address
					gCVAddress--;
				}
				
				//Display CV No.
				sprintf(gText, "%2d", gCVAddress);
				LCD.printBigNumber(gText, 7, 2);
				break;
				
			case 1: //CV Value select
				//Decrement CV no
				if(gColValue[gCol] == 0) {
					if(gCol == 0) {
						gColValue[0] = 2;
					} else if(gColValue[0] == 2) {
						gColValue[gCol] = 5;
					} else {
						gColValue[gCol] = 9;
					}
				} else {
					gColValue[gCol]--;
				}
				
				//Display CV Val.
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 4);
			}
			
			gPressCount = 0;
		}
		// Set CVValue
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			if(gUpDwnPos == 1)
			{
				LCD.drawIcon(10, 6, 2, false);
				LCD.drawIcon(11, 6, 2, false);
				
				//CV Write
				gCVValue = gColValue[0] * 100 + gColValue[1] * 10 + gColValue[2];
				DSCore.WriteConfig_Dir(gCVAddress, gCVValue);
				
				#ifdef _DEBUG
				Serial.print("CVW:");
				Serial.print(gCVAddress);
				Serial.print(",");
				Serial.println(gCVValue);
				#endif
				
				// clear "*"
				LCD.printString("   ", 6, 6);
				// clear "->"
				LCD.printString("  ", 10, 6);
				// clear [CVW]
				LCD.printString("   ", 10, 7);
				
				//Mode
				LCD.printString(" ", 5, 4);
				LCD.printString(" ", 5, 5);
				LCD.drawIcon(5, 2, 9, false);
				LCD.drawIcon(5, 3, 10, false);
				
				gCol = 0;
				gUpDwnPos = 0;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				switch(gUpDwnPos) {
				  case 0: //CV No select
					//Increment CV no
					gCVAddress++;
					if(gCVAddress > MAX_CV) {
						gCVAddress = 1;
					}
					
					//Display CV No.
					sprintf(gText, "%2d", gCVAddress);
					LCD.printBigNumber(gText, 7, 2);
					break;
					
				  case 1: //CV Value select
					//Increment CV value
					gColValue[gCol]++;
					if(gCol == 0) {
						if(gColValue[gCol] > 2) gColValue[0] = 0;
					}
					else {
						if(gColValue[0] == 2) {
							if(gColValue[gCol] > 5) gColValue[gCol] = 0;
						} else {
							if(gColValue[gCol] > 9) gColValue[gCol] = 0;
						}
					}
					
					//Display CV Val.
					sprintf(gText, "%1d", gColValue[gCol]);
					LCD.printBigNumber(gText, 6 + gCol, 4);
					break;
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				switch(gUpDwnPos) {
				  case 0: //CV No select
					//Decrement CV no
					if(gCVAddress == 1) {
						gCVAddress = MAX_CV;
					} else {
						//Decrement address
						gCVAddress--;
					}
					
					//Display CV No.
					sprintf(gText, "%2d", gCVAddress);
					LCD.printBigNumber(gText, 7, 2);
					break;
					
				  case 1: //CV Value select
					//Decrement CV no
					if(gColValue[gCol] == 0) {
						if(gCol == 0) {
							gColValue[0] = 2;
						} else if(gColValue[0] == 2) {
							gColValue[gCol] = 5;
						} else {
							gColValue[gCol] = 9;
						}
					} else {
						gColValue[gCol]--;
					}
					
					//Display CV Val.
					sprintf(gText, "%1d", gColValue[gCol]);
					LCD.printBigNumber(gText, 6+ gCol, 4);
				}
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*** Process Dial ***/
void processModeDial()
{
	// Dir
	static byte gLocDirection = GO_FWD;
	
	if(gButton_val != gButton_old)
	{
		if(checkButton(gButton_old, gButton_val, BUTTON_DIR))
		{
			gLocDirection = (gLocDirection == GO_FWD) ? GO_REV : GO_FWD;
			DSCore.SetLocoDirection(gLocProtcol + gLocAddress, gLocDirection);
			
			if(gAddrIdx != gAddrIdxOld) {
				//Update LocAddress
				saveEEPROM(2);
				
				gAddrIdxOld = gAddrIdx;
			}
			
			//Display Direction
			if(gLocDirection == GO_FWD) {
				LCD.printString(" ", 13, 1);
				LCD.drawIcon(17, 1, 11, false); // Fwd
			} else {
				LCD.drawIcon(13, 1, 12, false); // Rev
				LCD.printString(" ", 17, 1);
			}
		}
	}
	
	//Dial
	static byte aLocSpeed = 0;
	//static byte aLocStep = 0;
	int aVol_value = (analogRead(A0) & 0b1111111100); // max 1020
	byte aSpeed = aVol_value >> 2;
	byte aStep = aSpeed >> 1;
	
	if(aSpeed != aLocSpeed)
	//if(aStep != aLocSpeed)
	{
		#ifdef _DEBUGPORT
		digitalWrite(A3, HIGH);
		#endif
		
		if(gAddrIdx != gAddrIdxOld)
		{
			//Update LocAddress
			saveEEPROM(2);
			
			gAddrIdxOld = gAddrIdx;
		}
		
		// analog
		if(gLocAddress == 0)
		{
			if( gLocDirection == GO_FWD) {
				analogWrite(9, 255 - aSpeed);
				analogWrite(10, 255);
			} else {
				analogWrite(10, 255 - aSpeed);
				analogWrite(9, 255);
			}
			
			// display analog speed
			#ifdef _SPEEDBAR
			LCD.drawSpeedBar( 14, 1, aSpeed, false, false);
			#else
			sprintf(gText, "%3d", aSpeed);
			LCD.printString(gText, 14, 1);
			#endif
			
			#ifdef _DEBUG
			Serial.print("Dial:");
			Serial.println(aSpeed);
			#endif
		}
		
		// digital
		else
		{
			DSCore.SetLocoSpeedEx(gLocProtcol + gLocAddress, aVol_value, SPEEDSTEP_DCC127);
			
			// display speed
			#ifdef _SPEEDBAR
			LCD.drawSpeedBar( 14, 1, aSpeed, false, false);
			#else
			sprintf(gText, "%3d", aSpeed);
			LCD.printString(gText, 14, 1);
			#endif
			
			#ifdef _DEBUG
			Serial.print("Dial:");
			Serial.println(aSpeed>>1);
			#endif
		}
		
		aLocSpeed = aSpeed;
		//aLocSpeed = aStep;
		
		#ifdef _DEBUGPORT
		digitalWrite(A3, LOW);
		#endif
	}
}

/*** Process Selecter ***/
void processModeSel()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_SEL))
		{
			//Reset press count
			gPressCount = 0;
		}
		//Buttons have no difference from previous state
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(checkButton(gButton_old, gButton_val, BUTTON_UP))
			{
				if(gMode_Drive == DRV_LOC)
				{
					gUpDwnPosOld = gUpDwnPos;
					if(gUpDwnPosOld == 2) gUpDwnPos = 1;
					gCol = 0;
					gMode_Drive = DRV_ACC;
				}
				else
				{
					if((gUpDwnPos != gUpDwnPosOld) && (gUpDwnPosOld == 2)) {
						gUpDwnPos = 2;
					} else {
						gUpDwnPos = gUpDwnPosOld;
					}
					gMode_Drive = DRV_LOC;
				}
				
				gDualButton = true;
				
				displayDiffer();
				gPartialCount = 1;
				
				#ifdef _DEBUG
				Serial.print("SEL+UP:Old");
				Serial.print(gUpDwnPosOld);
				Serial.print(",Pos");
				Serial.println(gUpDwnPos);
				#endif
			}
			else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
			{
				if(gMode_Drive != DRV_ROUTE)
				{
					gMode_Drive = DRV_ROUTE;
					
					gUpDwnPosOld = gUpDwnPos;
					if(gUpDwnPosOld == 2) {
						gUpDwnPos = 1;
					}
					gRoutePos = gUpDwnPos;
					gPartialCount = 1;
				}
				else
				{
					// Route edit
					//gUpDwnPosOld = gUpDwnPos;
					gUpDwnPos = 2;
					gCol = 0;
					displayDiffer();
				}
				
				gDualButton = true;
				
				#ifdef _DEBUG
				Serial.print("SEL+DOWN:Old");
				Serial.print(gUpDwnPosOld);
				Serial.print(",Pos");
				Serial.println(gUpDwnPos);
				#endif
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButtonLeaved(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(gDualButton) {
				#ifdef _DEBUG
				Serial.print("DualButton:");
				Serial.println(gDualButton);
				#endif
				
				if(gMode_Drive == DRV_LOC) {
					gDualButton = false;
				}
				return;
			}
			
			if(gMode_Drive == DRV_LOC)
			{
				gUpDwnPos++;
				if(gUpDwnPos > 2) gUpDwnPos = 0;
			}
			else if(gMode_Drive == DRV_ACC)
			{
				gUpDwnPos++;
				if(gUpDwnPos == 1) {
					gDualButton = true;
				}
				if(gUpDwnPos > 1) gUpDwnPos = 1;
			}
			else if(gMode_Drive == DRV_ROUTE)
			{
				if(gUpDwnPos == 0) {
					gUpDwnPos = 1;
					gRoutePos = gUpDwnPos;
					gUpDwnPosOld = gUpDwnPos;
				} else if(gUpDwnPos == 1) {
					gUpDwnPos = 0;
					gRoutePos = gUpDwnPos;
					gUpDwnPosOld = gUpDwnPos;
				}
			}
			displayDiffer();
			
			//Reset press count
			gPressCount = 0;
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_SEL))
		{
			if( gPressCount > 15)
			{
				gDualButton = true;
				gPressCount = 0;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*** Process Loco ***/
void processModeLoc()
{
	static boolean bupdate = false;
	
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			if(gDualButton) return;
			
			#ifdef _DEBUGPORT
			digitalWrite(A3, HIGH);
			#endif
			//Increment address
			gAddrIdx++;
			if(gAddrIdx >= MAX_LOCO)
			{
				gAddrIdx = 0;
				
				TCCR0B &= B11111000; // B1,B2 PWM mode
				TCCR0B |= B00000011; // 1/64(500Hz)
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = true;
			}
			else
			{
				if(bAnalog) {
					TCCR0B = 3; // Normal mode, 1/64 prescal
					analogWrite(9, 0);
					analogWrite(10, 0);
					
					bAnalog = false;
				}
			}
			
			loadLongAddrEEP(gAddrIdx);
			
			//Display Loc Address
			sprintf(gText, "%4d", gLocAddress);
			LCD.printBigNumber(gText, 6, 0);
			
			//Display LocProtcol
			if(gAddrIdx == 0)
			{
				LCD.printString("Ana", 14, 0);
				bupdate = false;
			}
			else if(!bupdate)
			{
				if(gLocProtcol == ADDR_DCC) {
					LCD.printString("DCC", 14, 0);
				} else {
					LCD.printString("MM ", 14, 0);
				}
				bupdate = true;
			}
			
			#ifdef _DEBUGPORT
			digitalWrite(A3, LOW);
			#endif
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gDualButton) return;
			
			if(gAddrIdx == 0) {
				gAddrIdx = MAX_LOCO -1;
			} else {
				//Decrement address
				gAddrIdx--;
			}
			
			if(!bAnalog && gAddrIdx == 0)
			{
				TCCR0B &= B11111000; // B1,B2 PWM mode
				TCCR0B |= B00000011; // 1/64(500Hz)
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = true;
			}
			else if(bAnalog && gAddrIdx != 0)
			{
				TCCR0B = 3; // Normal mode, 1/64 prescal
				analogWrite(9, 0);
				analogWrite(10, 0);
				
				bAnalog = false;
			}
			
			loadLongAddrEEP(gAddrIdx);
			
			//Display Loc Address
			sprintf(gText, "%4d", gLocAddress);
			LCD.printBigNumber(gText, 6, 0);
			
			//Display LocProtcol
			if(gAddrIdx == 0) {
				LCD.printString("Ana", 14, 0);
				bupdate = false;
			} else if(!bupdate) {
				if(gLocProtcol == ADDR_DCC) {
					LCD.printString("DCC", 14, 0);
				} else {
					LCD.printString("MM ", 14, 0);
				}
				bupdate = true;
			}
			
			//Reset press count
			gPressCount = 0;
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment address
				gAddrIdx++;
				if(gAddrIdx >= MAX_LOCO) gAddrIdx = 1;
				
				loadLongAddrEEP(gAddrIdx);
				
				//Display Loc Address
				sprintf(gText, "%4d", gLocAddress);
				LCD.printBigNumber(gText, 6, 0);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				if(gAddrIdx == 0) {
					gAddrIdx = MAX_LOCO -1;
				} else {
					//Decrement address
					gAddrIdx--;
					if(gAddrIdx == 0) {
						gAddrIdx = MAX_LOCO -1;
					}
				}
				
				loadLongAddrEEP(gAddrIdx);
			
				//Display Loc No.
				sprintf(gText, "%4d", gLocAddress);
				LCD.printBigNumber(gText, 6, 0);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*** Process Function ***/
void processModeFunc()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			if(gDualButton) return;
			
			//Increment func number
			if(gUpDwnPos == 1)
			{
				gFunc1++;
				if(gFunc1 > MAX_FUNC) gFunc1 = 0;
			
				//Display Func No.
				sprintf(gText, "%2d", gFunc1);
				LCD.printBigNumber(gText, 6, 3);
				
				//Button
				LCD.printStringNOT("F", 10, 7);
				LCD.printStringNOT(gText, 11, 7);
			}
			else if(gUpDwnPos == 2)
			{
				gFunc2++;
				if(gFunc2 > MAX_FUNC) gFunc2 = 0;
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc2);
				LCD.printBigNumber(gText, 12, 3);
				
				//Button
				LCD.printStringNOT("F", 15, 7);
				LCD.printStringNOT(gText,16, 7);
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gUpDwnPos == 1)
			{
				if(gFunc1 == 0) {
					gFunc1 = MAX_FUNC;
				} else {
					gFunc1--;
				}
				//Display Func No.
				sprintf(gText, "%2d", gFunc1);
				LCD.printBigNumber(gText, 6, 3);
				
				//Button
				LCD.printStringNOT("F", 10, 7);
				LCD.printStringNOT(gText, 11, 7);
			}
			else if(gUpDwnPos == 2)
			{
				if(gFunc2 == 0) {
					gFunc2 = MAX_FUNC;
				} else {
					gFunc2--;
				}
				
				//Display Func No.
				sprintf(gText, "%2d", gFunc2);
				LCD.printBigNumber(gText, 12, 3);
				
				//Button
				LCD.printStringNOT("F", 15, 7);
				LCD.printStringNOT(gText,16, 7);
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			if(gLocAddress == 0) return;
			
			gLocFunc1 = (gLocFunc1 == 0) ? 1 : 0;
			if(gLocFunc1 == 1) {
				LCD.printString("On ", 6, 5);
			} else {
				LCD.printString("Off", 6, 5);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc1, gLocFunc1);
			
			if(gFunc1 != gFunc1Old) {
				//Update Func1
				saveEEPROM(4);
				gFunc1Old = gFunc1;
			}
			
			if(gAddrIdx != gAddrIdxOld) {
				saveEEPROM(2);
				gAddrIdxOld = gAddrIdx;
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			if(gLocAddress == 0) return;
			
			gLocFunc2 = (gLocFunc2 == 0) ? 1 : 0;
			if(gLocFunc2 == 1) {
				LCD.printString("On ", 12, 5);
			} else {
				LCD.printString("Off", 12, 5);
			}
			
			DSCore.SetLocoFunction(gLocProtcol + gLocAddress, gFunc2, gLocFunc2);
			
			if(gFunc2 != gFunc2Old) {
				//Update Func2
				saveEEPROM(5);
				gFunc2Old = gFunc2;
			}
			
			if(gAddrIdx != gAddrIdxOld) {
				saveEEPROM(2);
				gAddrIdxOld = gAddrIdx;
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment func number
				if(gUpDwnPos == 1)
				{
					gFunc1++;
					if(gFunc1 > MAX_FUNC) {
						gFunc1 = 0;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc1);
					LCD.printBigNumber(gText, 6, 3);
					
					//Button
					LCD.printStringNOT("F", 10, 7);
					LCD.printStringNOT(gText, 11, 7);
					
					gPressCount = CONTINUOUS;
				}
				else if(gUpDwnPos == 2)
				{
					gFunc2++;
					if(gFunc2 > MAX_FUNC) {
						gFunc2 = 0;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc2);
					LCD.printBigNumber(gText, 12, 3);
					
					//Button
					LCD.printStringNOT("F", 15, 7);
					LCD.printStringNOT(gText,16, 7);
					
					gPressCount = CONTINUOUS;
				}
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Decrement func number
				if(gUpDwnPos == 1)
				{
					if(gFunc1 == 0) {
						gFunc1 = MAX_FUNC;
					} else {
						gFunc1--;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc1);
					LCD.printBigNumber(gText, 6, 3);
					
					//Button
					LCD.printStringNOT("F", 10, 7);
					LCD.printStringNOT(gText, 11, 7);
					
					gPressCount = CONTINUOUS;
				}
				else if(gUpDwnPos == 2)
				{
					if(gFunc2 == 0) {
						gFunc2 = MAX_FUNC;
					} else {
						gFunc2--;
					}
					
					//Display Func No.
					sprintf(gText, "%2d", gFunc2);
					LCD.printBigNumber(gText, 12, 3);
					
					//Button
					LCD.printStringNOT("F", 15, 7);
					LCD.printStringNOT(gText,16, 7);
					
					gPressCount = CONTINUOUS;
				}
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*** Process Accessory ***/
void processModeAcc()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButtonLeaved(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(gDualButton) {
				#ifdef _DEBUG
				Serial.println("Acc DualButton");
				#endif
				
				gDualButton = false;
				return;
			}
			
			if(gUpDwnPos == 1)
			{
				gCol++;
				if(gCol > 3) {
					gUpDwnPos = 0;
					gCol = 0;
					gUpDwnPosOld = gUpDwnPos;
					displayDiffer();
				} else {
					LCD.printString(" ", 5 + gCol, 5);
					LCD.printString("*", 6 + gCol, 5);
				}
			}
			
			#ifdef _DEBUG
			Serial.print("Acc SEL:");
			Serial.print(gUpDwnPos);
			Serial.print(",");
			Serial.println(gCol);
			#endif
		}
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			if(gDualButton) return;
			
			//Increment address
			if(gUpDwnPos == 1)
			{
				gColValue[gCol]++;
				if(gCol == 0) {
					if(gColValue[0] > 2) gColValue[0] = 0;
				}
				else {
					if(gColValue[0] == 2) {
						if(gCol == 1) {
							gColValue[1] = 0;
						} else if((gCol == 2) || (gCol == 3)) {
							if(gColValue[gCol] > 4) {
								gColValue[gCol] = 0;
							}
						}
					}
					else {
						if(gColValue[gCol] > 9) gColValue[gCol] = 0;
					}
				}
				
				//colum
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 3);
				
				#ifdef _DEBUG
				Serial.print("Acc UP:");
				Serial.print(gColValue[gCol]);
				Serial.print("[");
				Serial.print(gCol);
				Serial.println("]");
				#endif
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			//Decrement address colum
			if(gUpDwnPos == 1)
			{
				if(gColValue[gCol] == 0)
				{
					if(gCol == 0) {
						gColValue[0] = 2;
					} else if(gColValue[0] == 2) {
						if(gCol == 1) {
							gColValue[1] = 0;
						} else if((gCol == 2) || (gCol == 3)) {
							gColValue[gCol] = 4;
						}
					} else {
						gColValue[gCol] = 9;
					}
				}
				else
				{
					gColValue[gCol]--;
				}
				
				//colum
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 3);
				
				#ifdef _DEBUG
				Serial.print("Acc DOWN:");
				Serial.print(gCol);
				Serial.print(",");
				Serial.print(gColValue[0]);
				Serial.print(gColValue[1]);
				Serial.print(gColValue[2]);
				Serial.println(gColValue[3]);
				#endif
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			//Diverse
			gAccAddr = (gColValue[0] * 1000 + gColValue[1] * 100
						+ gColValue[2] * 10 + gColValue[3]);
			if(gAccAddr > 0)
			{
				if(eepProtcol == DCC_MODE)
				{
					if(gAccAddr > MAX_ACC) {
						gAccAddr = 1;
					} else {
						DSCore.SetTurnout(gAccProtcol + gAccAddr, 0);
						//Acc Message
						LCD.printString("Div/", 14, 4);
						//Update DCC Acc
						if(gAccAddr != gAccAddrOld) {
							saveEEPROM(6);
							gAccAddrOld = gAccAddr;
						}
					}
				}
				else if(eepProtcol == MM_MODE)
				{
					if(gAccAddr > MAX_MMACC) {
						gAccAddr = 1;
					} else {
						//Acc Message
						LCD.printString("Div/", 14, 4);
						//Update MM Acc
						if(gAccAddr != gAccAddrOld) {
							saveEEPROM(8);
							gAccAddrOld = gAccAddr;
						}
					}
				}
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			//Straight
			gAccAddr = (gColValue[0] * 1000 + gColValue[1] * 100
						+ gColValue[2] * 10 + gColValue[3]);
			
			if(gAccAddr > 0)
			{
				if(eepProtcol == DCC_MODE)
				{
					if(gAccAddr > MAX_ACC) {
						gAccAddr = 0;
					} else {
						DSCore.SetTurnout(gAccProtcol + gAccAddr, 1);
						//Acc Message
						LCD.printString("Str|", 14, 4);
						//Update DCC Acc
						if(gAccAddr != gAccAddrOld) {
							saveEEPROM(6);
							gAccAddrOld = gAccAddr;
						}
					}
				}
				else if(eepProtcol == MM_MODE)
				{
					if(gAccAddr > MAX_MMACC) {
						gAccAddr = 0;
					} else {
						DSCore.SetTurnout(gAccProtcol + gAccAddr, 1);
						//Acc Message
						LCD.printString("Str|", 14, 4);
						//Update MM Acc
						if(gAccAddr != gAccAddrOld)
						{
							saveEEPROM(8);
							gAccAddrOld = gAccAddr;
						}
					}
				}
			}
		}
	}
	else
	{
		//Buttons have no difference from previous state
		if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_UP))
		{
			if( gPressCount > CONTINUOUS)
			{
				//Increment address
				if(gUpDwnPos == 1)
				{
					gColValue[gCol]++;
					if(gCol == 0) {
						if(gColValue[0] > 2) gColValue[0] = 0;
					}
					else {
						if(gColValue[0] == 2) {
							if(gCol == 1) {
								gColValue[1] = 0;
							} else if((gCol == 2) || (gCol == 3)) {
								if(gColValue[gCol] > 4) {
									gColValue[gCol] = 0;
								}
							}
						} else {
							if(gColValue[gCol] > 9) gColValue[gCol] = 0;
						}
					}
					
					//colum
					sprintf(gText, "%1d", gColValue[gCol]);
					LCD.printBigNumber(gText, 6 + gCol, 3);
					
					gPressCount = CONTINUOUS;
				}
			}
			else
			{
				gPressCount++;
			}
		}
		else if(checkButtonContinuous(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if( gPressCount > CONTINUOUS)
			{
				if(gColValue[gCol] == 0)
				{
					if(gCol == 0) {
						gColValue[0] = 2;
					} else if(gColValue[0] == 2) {
						if(gCol == 1) {
							gColValue[1] = 0;
						} else if((gCol == 2) || (gCol == 3)) {
							gColValue[gCol] = 4;
						}
					} else {
						gColValue[gCol] = 9;
					}
				}
				else
				{
					gColValue[gCol]--;
				}
				
				//colum
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printBigNumber(gText, 6 + gCol, 3);
				
				gPressCount = CONTINUOUS;
			}
			else
			{
				gPressCount++;
			}
		}
	}
}

/*** Process Route ***/
void processModeRoute()
{
	if(gButton_val != gButton_old)
	{
		//Buttons have difference from previous state
		if(checkButtonLeaved(gButton_old, gButton_val, BUTTON_SEL))
		{
			if(gDualButton) {
				gDualButton = false;
				return;
			}
			
			switch(gUpDwnPos) {
			  case 2:
				gUpDwnPos = 3;
				gCol = 0;
				LCD.printString(" ", 12, 4);
				LCD.printString("*", 14, 4);
				break;
				
			  case 3:
				gCol++;
				if(gCol > 3) {
					gCol = 0;
					gUpDwnPos = 2;
					LCD.printString(" ", 17, 4);
					LCD.printString("*", 12, 4);
				} else {
					LCD.printString(" ", 13 + gCol, 4);
					LCD.printString("*", 14 + gCol, 4);
				}
			}
			displayDiffer();
			
			#ifdef _DEBUG
			Serial.print("Route SEL:Old");
			Serial.print(gUpDwnPosOld);
			Serial.print(",Pos");
			Serial.print(gUpDwnPos);
			Serial.print(",Col");
			Serial.println(gCol);
			#endif
		}
		if(checkButton(gButton_old, gButton_val, BUTTON_UP))
		{
			//Increment route number
			switch(gUpDwnPos) {
			  case 1: // Route number select
				gRouteNoOld = gRouteNo;
				gRouteNo++;
				if(gRouteNo > MAX_ROUTE) gRouteNo = 1;
				
				sprintf(gText, "%2d", gRouteNo);
				LCD.printBigNumber(gText, 6, 3);
				// top of AccList addr
				gAccAddr = loadRouteAddrEEP(gRouteNo, 1);
				displayStringsList(14, 3, gAccAddr, false);
				break;
				
			  case 2: // Acc list select
				if(gAccAddr > 0) {
					gAccList++;
					sprintf(gText, "%1d", gAccList);
					LCD.printString(gText, 12, 3);
				}
				// AccList addr
				gAccAddr = loadRouteAddrEEP(gRouteNo, gAccList);
				displayStringsList(14, 3, gAccAddr, false);
				
				#ifdef _DEBUG
				Serial.print("Select list:");
				Serial.println(gAccList);
				#endif
				break;
				
			  case 3: // Edit mode
				gColValue[gCol]++;
				if(gCol == 0) {
					if(gColValue[0] > 2) gColValue[0] = 0;
				}
				else {
					if(gColValue[0] == 2) {
						if(gCol == 1) {
							gColValue[1] = 0;
						} else if((gCol == 2) || (gCol == 3)) {
							if(gColValue[gCol] > 4) {
								gColValue[gCol] = 0;
							}
						}
					}
					else {
						if(gColValue[gCol] > 9) gColValue[gCol] = 0;
					}
				}
				//colum
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printString(gText, 14 + gCol, 3);
				
				#ifdef _DEBUG
				Serial.print("Route edit:");
				Serial.print(gColValue[gCol]);
				Serial.print("[");
				Serial.print(gCol);
				Serial.println("]");
				#endif
				break;
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_DOWN))
		{
			if(gDualButton) return;
			
			switch(gUpDwnPos) {
			  case 1: // Route number select
				gRouteNoOld = gRouteNo;
				if(gRouteNo == 1) {
					gRouteNo = MAX_ROUTE;
				} else {
					gRouteNo--;
				}
				
				sprintf(gText, "%2d", gRouteNo);
				LCD.printBigNumber(gText, 6, 3);
				// top of AccList addr
				gAccAddr = loadRouteAddrEEP(gRouteNo, 1);
				displayStringsList(14, 3, gAccAddr, false);
				break;
				
			  case 2: // Acc list select
				if(gAccList > 1) gAccList--;
				
				sprintf(gText, "%1d", gAccList);
				LCD.printString(gText, 12, 3);
				// AccList addr
				gAccAddr = loadRouteAddrEEP(gRouteNo, gAccList);
				displayStringsList(14, 3, gAccAddr, false);
				break;
				
			  case 3: // Edit mode
				if(gColValue[gCol] == 0)
				{
					if(gCol == 0) {
						gColValue[0] = 2;
					} else if(gColValue[0] = 2) {
						if(gCol == 1) {
							gColValue[1] = 0;
						} else if((gCol == 2) || (gCol == 3)) {
							gColValue[gCol] = 4;
						}
					} else {
						gColValue[gCol] = 9;
					}
				}
				else
				{
					gColValue[gCol]--;
				}
				//colum
				sprintf(gText, "%1d", gColValue[gCol]);
				LCD.printString(gText, 14 + gCol, 3);
				break;
			}
			
			//Reset press count
			gPressCount = 0;
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F1))
		{
			//Diverse
			if(gUpDwnPos < 2)
			{
				gAccSwitch = 0;
				gRouteCount = 1;
				
				//Acc Message
				LCD.printString("Div/", 12, 5);
				
				if(gRouteNo != gRouteNoOld)
				{
					//Update Route
					saveEEPROM(10);
					gRouteNoOld = gRouteNo;
				}
			}
			// Route update
			else
			{
				LCD.printString("****", 14, 4);
				
				gAccAddr = (gColValue[0] * 1000 + gColValue[1] * 100
							+ gColValue[2] * 10 + gColValue[3]);
				
				if(eepProtcol == DCC_MODE) {
					if(gAccAddr <= MAX_ACC) {
						saveRouteAddrEEP(gRouteNo, gAccList, gAccAddr);
					}
				} else if(eepProtcol == MM_MODE) {
					if(gAccAddr <= MAX_MMACC) {
						saveRouteAddrEEP(gRouteNo, gAccList, gAccAddr);
					}
				}
				
				LCD.printString("    ", 14, 4);
				
				// select AccList No cursol
				gCol = 0;
				gUpDwnPos = 2;
				displayDiffer();
			}
		}
		else if(checkButton(gButton_old, gButton_val, BUTTON_F2))
		{
			//Straight
			if(gUpDwnPos < 2)
			{
				gAccSwitch = 1;
				gRouteCount = 1;
				
				//Acc Message
				LCD.printString("Str|", 12, 5);
				
				if(gRouteNo != gRouteNoOld)
				{
					//Update Route
					saveEEPROM(10);
					gRouteNoOld = gRouteNo;
				}
			}
			// ESC (exit edit)
			else
			{
				// top of AccList addr
				gAccAddr = loadRouteAddrEEP(gRouteNo, 1);
				displayStringsList(14, 3, gAccAddr, false);
				gAccList = 1;
				LCD.printString("adr1", 9, 3);
				gUpDwnPos = gRoutePos;
				displayDiffer();
			}
			
			#ifdef _DEBUG
			Serial.print("ESC:Old");
			Serial.print(gUpDwnPosOld);
			Serial.print(",Pos");
			Serial.println(gUpDwnPos);
			#endif
		}
	}
}

/*** Process Send Route ***/
void processSendRoute()
{
	static byte aInterval = 0;
	
	aInterval++;
	if(aInterval > 5) // 500msec
	{
		aInterval = 0;
		
		word aAddr = loadRouteAddrEEP(gRouteNo, gRouteCount);
		if(aAddr == 0) {
			gRouteCount = 0;
		} else {
			DSCore.SetTurnout(gAccProtcol + aAddr, gAccSwitch);
			
			#ifdef _DEBUG
			Serial.print("SendRoute:");
			Serial.print(gRouteNo);
			Serial.print(",");
			Serial.println(aAddr);
			#endif
			gRouteCount++;
		}
		
		if(gRouteCount > MAX_RTADR) {
			gRouteCount = 0;
		}
	}
}

/*******************************
    Make Strings from ColValue
********************************/

void mkColValArray(byte *inArray, word inData)
{
	word mod = inData % 1000;
	inArray[0] = inData / 1000;
	sprintf(&gText[0], "%1d", inArray[0]);
	inArray[1] = mod / 100;
	sprintf(&gText[2], "%1d", inArray[1]);
	mod = mod % 100;
	inArray[2] = mod / 10;
	sprintf(&gText[4], "%1d", inArray[2]);
	inArray[3] = mod % 10;
	sprintf(&gText[6], "%1d", inArray[3]);
	
	#ifdef _DEBUG
	Serial.print("mkColValArray:");
	Serial.print(inArray[0]);
	Serial.print(",");
	Serial.print(inArray[1]);
	Serial.print(",");
	Serial.print(inArray[2]);
	Serial.print(",");
	Serial.print(inArray[3]);
	Serial.print(",");
	Serial.println(gText);
	#endif
}

/*******************************
    ButtonStatus
********************************/

/*** Get button status ***/
byte getButtonStatus()
{
	byte aButton_val = 0;
	
	//Input Buttons
	aButton_val = PIND & 0b11111100;
	aButton_val ^= 0x80;
	
	return ~aButton_val;
}

/*** Check button ***/
bool checkButton(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( ((inBuf & inButtonBit) == 0) && ((inCurrent & inButtonBit) > 0))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*** Check button leaved ***/
bool checkButtonLeaved(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( ((inBuf & inButtonBit) > 0) && ((inCurrent & inButtonBit) == 0))
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*** Check button continuous ***/
bool checkButtonContinuous(byte inBuf, byte inCurrent, byte inButtonBit)
{
	if( (inCurrent & inButtonBit) > 0)
	{
		return true;
	}
	else
	{
		return false;
	}
}

/*******************************
    EEPROM
********************************/

/*** Load eeprom ***/
void loadEEPROM()
{
	if( EEPROM.read(0) == 0xCD)
	{
		gAddrIdx = EEPROM.read(2);
		if(gAddrIdx >= MAX_LOCO) gAddrIdx = 3;
		
		eepProtcol = EEPROM.read(3);
		if( eepProtcol > MM_MODE)
		{
			eepProtcol = DCC_MODE;
		}
		
		gFunc1 = EEPROM.read(4);
		if(gFunc1 > MAX_FUNC) gFunc1 = 0;
		
		gFunc2 = EEPROM.read(5);
		if(gFunc2 > MAX_FUNC) gFunc2 = 1;
		
		if(eepProtcol == DCC_MODE) {
			gAccAddr = (EEPROM.read(7) << 8) + EEPROM.read(6);
		} else {
			gAccAddr = (EEPROM.read(9) << 8) + EEPROM.read(8);
		}
		gRouteNo = EEPROM.read(10);
		
		#ifdef _DEBUG
		Serial.println("");
		Serial.println("loadEEPROM():");
		
		Serial.print("Protocol=");
		Serial.print(eepProtcol);
		Serial.print(".AddrIdx=");
		Serial.print(gAddrIdx);
		Serial.print(".Func1=");
		Serial.print(gFunc1);
		Serial.print(",Func2=");
		Serial.print(gFunc2);
		
		Serial.print(",AccAddr=");
		Serial.print(gAccAddr);
		Serial.print(",RouteNo=");
		Serial.println(gRouteNo);
		#endif
	}
}

/*** Save eeprom ***/
void saveEEPROM(byte inItem)
{
	if( EEPROM.read(0) == 0xCD)
	{
		switch(inItem) {
		  case 2:
			EEPROM.write(2, gAddrIdx);
			break;
		  case 3:
			EEPROM.write(3, eepProtcol);
			break;
		  case 4:
			EEPROM.write(4, gFunc1);
			break;
		  case 5:
			EEPROM.write(5, gFunc2);
			break;
		  case 6:
			EEPROM.write(6, gAccAddr % 256);
			EEPROM.write(7, gAccAddr / 256);
			break;
		  case 8:
			EEPROM.write(8, gAccAddr % 256);
			EEPROM.write(9, gAccAddr / 256);
			break;
		  case 10:
			EEPROM.write(10, gRouteNo);
			break;
		}
		
		#ifdef _DEBUG
		Serial.print("saveEEP:");
		Serial.println(inItem);
		#endif
	}
	else
	{
		//Data write
		EEPROM.write(1, 0);
		EEPROM.write(2, gAddrIdx);
		EEPROM.write(3, eepProtcol);
		
		EEPROM.write(4, gFunc1);
		EEPROM.write(5, gFunc2);
		
		EEPROM.write(6, gAccAddr % 256);
		EEPROM.write(7, gAccAddr / 256);
		
		if(eepProtcol == MM_MODE) {
			if(gAccAddr > MAX_MMACC) gAccAddr = MAX_MMACC;
			EEPROM.write(8, gAccAddr % 256);
			EEPROM.write(9, gAccAddr / 256);
		}
		
		EEPROM.write(10, gRouteNo);
	}
}

/*** load long address ***/
void loadLongAddrEEP(byte inIdx)
{
	if(inIdx == 0)
	{
		gLocAddress = 0;
		return;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	int eepaddr = EEPOFFSET + (inIdx -1) * 2;
	
	if( EEPROM.read(0) == 0xCD)
	{
		// about 10usec
		gLocAddress = (EEPROM.read(eepaddr + 1) << 8) + EEPROM.read(eepaddr);
		/*if(gLocAddress == 0) {
			gLocAddress = 1;
		} else if(gLocAddress > 9999) {
			gLocAddress = 9999;
		}*/
		
		#ifdef _DEBUG
		Serial.print("longAddress:");
		Serial.println(gLocAddress);
		#endif
	}
	else
	{
		// init EEPROM
		saveEEPROM(2);
		
		// init RouteAddr
		loadRouteAddrEEP(gRouteNo, 1);
		
		//Mint can ID
		EEPROM.write(0, 0xCD);
		
		for(int i=1; i<=MAX_LOCO; i++)
		{
			saveLongAddrEEP(i, i);
		}
		
		eepaddr = EEPOFFSET;
		gLocAddress = (EEPROM.read(eepaddr + 1) << 8) + EEPROM.read(eepaddr);
		if(gLocAddress > 9999) gLocAddress = 3;
	}
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	#endif
}

/*** save long address ***/
void saveLongAddrEEP(byte inIdx, word inAddr)
{
	if(inIdx == 0) return;
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, HIGH);
	#endif
	
	int eepaddr = EEPOFFSET + (inIdx -1) * 2;
	
	EEPROM.write(eepaddr, lowByte(inAddr));
	EEPROM.write(eepaddr + 1, highByte(inAddr));
	EEPROM.write(2, inIdx);
	
	#ifdef _DEBUGPORT
	digitalWrite(A3, LOW);
	
	Serial.print("SaveLAddr:");
	Serial.print(inAddr);
	Serial.print("[");
	Serial.print(inIdx);
	Serial.println("]");
	#endif
}

/*** load route address ***/
word loadRouteAddrEEP(byte inRout, byte inAccList)
{
	int eepaddr;
	word aAddr = 1;
	
	if( EEPROM.read(0) == 0xCD)
	{
		eepaddr = EEPROUTE + (inRout -1) *20 + (inAccList -1) *2;
		aAddr = (EEPROM.read(eepaddr +1) << 8) + EEPROM.read(eepaddr);
		
		if(eepProtcol == DCC_MODE) {
			if(aAddr > MAX_ACC) aAddr = 0;
		} else if(eepProtcol == MM_MODE) {
			if(aAddr > MAX_MMACC) aAddr = 0;
		}
		
		#ifdef _DEBUG
		Serial.print("loadRouteAddrEEP:");
		Serial.print(eepaddr);
		Serial.print("=");
		Serial.println(aAddr);
		#endif
	}
	else
	{
		for(int i=0; i<MAX_ROUTE; i++) {
			eepaddr = EEPROUTE + i * 20;
			EEPROM.write(eepaddr, i+1);
			EEPROM.write(eepaddr +1, 0);
		}
	}
	
	return aAddr;
}

/*** save route address ***/
void saveRouteAddrEEP(byte inRout, byte inAccList, word inAccAddr)
{
	int eepaddr = EEPROUTE + (inRout -1) *20 + (inAccList -1) *2;
	
	EEPROM.write(eepaddr +1, inAccAddr / 256);
	EEPROM.write(eepaddr, inAccAddr % 256);
	
	#ifdef _DEBUG
	Serial.print("saveRouteAddrEEP:");
	Serial.print(eepaddr);
	Serial.print("=");
	Serial.println(inAccAddr);
	#endif
}
